#!/bin/bash
# Copyright 2005 - 2013, Intel Corporation, All Rights Reserved.
# This script uninstalls Intel(R) Composer XE 2013 for Linux* and related products.

function ABS_PATH() {

    local path="$1"
    local cwd

    if [[ "${path:0:1}" != "/" ]]; then
        cwd=$( pwd ) || DIE "Cannot get name of current directory (?!)."
        if [[ "${cwd:-1:1}" == "/" ]]; then
            path="$cwd$path"
        else
            path="$cwd/$path"
        fi
    fi
    RS="$path"

} # function abs_path

function REL_TO_ABS(){
    if echo "$1" | egrep -e ";" -e "'" &>/dev/null; then
        return 1
    fi

    local rv=
    eval local path=$1
    local link_skip_mode="$2"
    local global_flag=
    local resolved_link=

    [ "x$link_skip_mode" == "x" ] && link_skip_mode=0

    local IS_VALID=`echo "$path" | grep "^/"`
    if [ -z "$IS_VALID" ]; then
        path=$(pwd)"/$path"
    fi    

    path=$(echo "$path" | sed 's/\/\/*/\//g')

    local last_dir="$(basename "$path")"
    local prev_dir=$(dirname "$path")
    if [ ! -d "$path" ]; then
        if [ ! -d "$prev_dir" ]; then
            REL_TO_ABS "$prev_dir"
            last_dir=$(echo "$last_dir" | sed 's/\/*$//')
	    RS="$RS/$last_dir"
        else
            last_dir=$(echo "$last_dir" | sed 's/\/*$//')
            GET_PATH_TO_DIRECTORY "$prev_dir"
            if [ "$RS" = "/" ]; then
                RS="$RS$last_dir"
            else
                RS="$RS/$last_dir"
            fi
            rv=0
        fi
    else
        GET_PATH_TO_DIRECTORY "$path"        
        rv=$?
    fi

    if [ "$rv" = "0" ] && [ "$link_skip_mode" == "0" ]; then
        while [ 0 ]; do
	    if [[ -L "$RS" ]]; then
		resolved_link=$(readlink "$RS")
		global_flag=$(echo "$resolved_link" | grep "^/" 2>/dev/null)
		if [ -z "$global_flag" ]; then
		    RS="$(dirname "$RS")/$resolved_link"
		else
		    RS="$resolved_link"
		fi
	    else
		break
	    fi
	done
        prev_dir=$(dirname "$RS")
        last_dir=$(basename "$RS")
        while [ "$prev_dir" != "/" ]; do
    	    if [[ -L "$prev_dir" ]]; then
		resolved_link=$(readlink "$prev_dir")
		global_flag=$(echo "$resolved_link" | grep "^/" 2>/dev/null)
		if [ -z "$global_flag" ]; then
		    prev_dir="$(dirname "$prev_dir")/$resolved_link"
		else
		    prev_dir="$resolved_link"
		fi
	    fi
            last_dir="$(basename "$prev_dir")/$last_dir"
            prev_dir=$(dirname "$prev_dir")
        done
        RS="$prev_dir$last_dir"
	REL_TO_ABS "$RS" 1
    fi
    return $rv
}

function GET_PATH_TO_DIRECTORY() {
    local path="$1"
    local prev_dir=
    local last_dir=
    (cd "$path" &>/dev/null)
     if [ "$?" = "0" ]; then
         path=$(cd "$path" &>/dev/null; pwd)
     else
         prev_dir=$(dirname "$path")
         last_dir=$(basename "$path")
         local flag=0;
         while [ "$flag" != "1" ]; do
             (cd "$prev_dir" &>/dev/null)
             [[ "$?" = "0" ]] && path="$prev_dir/$last_dir" && flag=1
             last_dir="$(basename "$prev_dir")/$last_dir"
             prev_dir=$(dirname "$prev_dir")
         done  
     fi
     if [ "$path" != "/" ]; then
         RS="$path"
         return 0
     else
         return 1
     fi
}

function RPM_INIT(){
	[ ${RPM_CHECK} -eq 0 ] || return ${ERR_OK}
	rpm -q rpm &> /dev/null

	
	if [ $? -ne 0 ] ; then
		LOG "Cannot get shared lock on RPM Database"
		return ${ERR_RPM_LOCK}
	fi

	RPM_CHECK=1
	
	return ${ERR_OK}
} # RPM_INIT(){   

function SPLIT() {

    local delimiter="$1"
    local string="$2"
    local ifs="$IFS"
    local -a result

    [[ $# -eq 2 ]] || CROAK "$FUNCNAME() expects 2 arguments."
    [[ "${#delimiter}" -eq 1 ]] || CROAK "$FUNCNAME(): The 1st argument must be a single char."

    IFS="$delimiter"; result=($string); IFS="$ifs"
    RA=( "${result[@]}" )

} # function split

function RPM_INFO(){
	local rpm_name=$1
	local rpm_tag=$2
	
	[ $RPM_CHECK -eq 1 ] || RPM_INIT ; local err=$?
	[ $err -eq ${ERR_OK} ] || return $err
	
	local opt="-q"
	if [ -f "$rpm_name" ]; then
		LOG "Using file query for package - '$rpm_name'"
		opt="-qp"
	fi #if[ -x $rpm_name ]; then
	
	local rpm_out_count=`rpm $opt $rpm_name 2>&1 | wc -l`
	if [ $rpm_out_count -eq 1 ] ; then 
		local info=`rpm $opt $rpm_name --qf %{$rpm_tag} 2>&1`
		if ! echo $info | grep installed &>/dev/null ; then
			echo $info
			LOG "Package search: '$rpm_name' => '$rpm_tag' => '$info' "
			return $ERR_OK;
		fi #if ! echo $info | grep installed &>/dv/null ; then
	else
		local out=
                local i=1
			local info=`rpm $opt $rpm_name --qf %{$rpm_tag}"\n" 2>&1 | sed $i'!d'`
			out="$out$info"
                echo $info
		LOG "Multiple package search: '$rpm_name' => '$rpm_tag' => '$out' "
		return ${ERR_OK}
	fi

	LOG "Package '$rpm_name' not found"
	
	return ${ERR_RPM_NO_PACKAGE}
} # RPM_INFO(){

function READ_DEFAULT_ANSWER(){	
	local param=$1
        local non_verbose=$2
	local retvalue=''
	LOG "silent:read_default_answer(): Trying to find the parameter '$param'"
	CONFIG_GET_VALUE $CONFIG_PREFIX $CONFIG_SECTION $param
	retvalue=$RS
	if [ "x$retvalue" == "x" ]; then
	    [[ "$non_verbose" = "1" ]] || WARN "Parameter \"$param\" was not found"
	    RS=''
	else
	    RS=$retvalue
	fi
} #read_default_answer(){
			                                                                                
function SET_DEFAULT_ANSWER(){	
	local question="$1"
	local value="$2"
	
	LOG "Set default answer for:"
	LOG_ARGS "$@"
	if IS_COMMAND_LINE_OPTION_EXISTS silent; then
	    LOG "Failed, silent mode does not support dynamic default answers."
	else
	    CONFIG_SET_VALUE $CONFIG_PREFIX $CONFIG_SECTION $question "$value"
	fi
} #read_default_answer(){

function SET_CONFIG_FILE(){
	if IS_COMMAND_LINE_OPTION_EXISTS silent; then
    	    CONFIG_FILENAME=`GET_COMMAND_LINE_OPTION_VALUE silent` || return 1
	    ABS_PATH $CONFIG_FILENAME
	    CONFIG_FILENAME=$RS
	    CONFIG_READ_FILE $CONFIG_PREFIX $CONFIG_FILENAME
	    LOG "CONFIG is set to '$CONFIG_FILENAME'"
	else
	    LOG "Silent mode was not enabled"
	fi	
} # set_config_file()

function KEEP_ANSWERS(){
	if IS_COMMAND_LINE_OPTION_EXISTS duplicate; then
    	    local dup_filename=`GET_COMMAND_LINE_OPTION_VALUE duplicate`
	    ABS_PATH $dup_filename
	    dup_filename=$RS
	    CONFIG_WRITE_FILE $CONFIG_PREFIX $dup_filename
	    LOG "Duplicate was performed into '$dup_filename'"
	else
	    LOG "Duplicate mode was not enabled"
	fi
}

function IS_ONE_OF() {

    [[ $# -ge 1 ]] || CROAK "$FUNCNAME() expects at least one argument."

    local item="$1"
    shift 1

    local iter
    for iter in "$@"; do
        if [[ "$item" == "$iter" ]]; then
            return 0
        fi
    done
    return 1

} # is_one_of

function GET_USER_INTERACTIVE_ANSWER() {
	local question_id=$1
    local question="$2"
    local USE_READLINE="$3"
	local l_answer=

	LOG "interact:get_user_interactive_answer(): getting answer on the question with id '$question_id'"
	if IS_COMMAND_LINE_OPTION_EXISTS silent ; then
		READ_DEFAULT_ANSWER $question_id
		l_answer=$RS
	else
    		if [ "x$USE_READLINE" != "xNO" ]; then
                    read -e -p "$question" l_answer
                else
                    read -p "$question" l_answer   
                fi
                if [ "${l_answer}x" == "x" ] ; then 	# user has pressed <Enter>
		    LOG "${0##*/}:$LINENO: interact:get_user_interactive_answer(): user entered empty answer"
		    LOG "${0##*/}:$LINENO: interact:get_user_interactive_answer(): using default answer"
		    READ_DEFAULT_ANSWER $question_id
		    l_answer=$RS
		fi
	fi
	LOG "interact:get_user_interactive_answer(): the answer is '$l_answer'"
	RS=$l_answer
} # get_user_interactive_answer

function READ_YES_NO_ANSWER ()
{
    local QUESTION=$1
    local question_id=$2
    local YES_ANSWER="y yes yea yeah yep ok"	# these answers are recognized as 'Yes'
    local NO_ANSWER="n no not nop nope"		# these answers are recognized as 'No'
    local DEFAULT_ANSWER=

    if ! IS_COMMAND_LINE_OPTION_EXISTS silent; then
		READ_DEFAULT_ANSWER $question_id
		DEFAULT_ANSWER=$RS
    fi

    while [ 1 -eq 1 ] ; do
        IS_COMMAND_LINE_OPTION_EXISTS silent || DISPLAY_QUESTION "" "$QUESTION" "Yes/No" "$DEFAULT_ANSWER"
        GET_USER_INTERACTIVE_ANSWER $question_id
        yesno="$(echo $RS | tr -s A-Z a-z)"
        if FIND_STRING $yesno ";" "$YES_ANSWER";then
	    LOG "interact: READ_YES_NO_ANSWER (): the answer is Yes"
            RS="yes"
            return 0
        else
            if FIND_STRING $yesno ";" "$NO_ANSWER";then
		LOG "interact: READ_YES_NO_ANSWER (): the answer is No"
                RS="no"
                return 1
            fi
	    if IS_COMMAND_LINE_OPTION_EXISTS silent;then
		LOG "interact: READ_YES_NO_ANSWER (): couldn't recognize the answer: config file parameter '$question_id' has wrong value '$answer'"
		return 2
	    fi
        fi
    done
}

function WAIT_ENTER_KEY ()
{
    IS_COMMAND_LINE_OPTION_EXISTS silent || read
}

function SAY() {
    local opts=""
    local log="log"
    while [[ "$1" == -[a-zA-Z] ]]; do
        case "$1" in
            -L ) log="";;
            *  ) opts="$opt $1";;
        esac
        shift 1
    done
    [[ "$log" != "" ]] && LOG_ARGS "$@"
    local message
    for message in "$@"; do
        echo $opts "$message"
    done
} # function say

function TELL() {
    LOG_ARGS "$@"
    set_tell_fmt
    if [ "x$_TELL_FMT_" != "x" ]; then
        echo "$*" | "$_TELL_FMT_" -s -w "$TELL_WIDTH"
    else
        echo "$*"
    fi
} # function tell

function set_tell_fmt() {
	[ "x$_SET_FMT_" == "xset" ] && return 0
	RUN which fmt &>/dev/null
	if [ $? == 0 ]; then 
		declare -r _TELL_FMT_="fmt" 
	else 
		declare -r _TELL_FMT_=
		WARN "Unable to find command \"fmt\". Some messages may be too long." 
	fi
	_SET_FMT_="set"
}

function ASK() {

    [[ $# -ge 2 && $# -le 4 ]] || CROAK "$FUNCNAME() expects from 2 to 4 arguments."
    LOG_ARGS "$@"

    local text="$1"
    local prompt="$2"      # Question itself.
    local options="$3"     # List of possible answers, delimited with '/'.
    local default="$4"     # Default answer to be used if user pressed Enter immediately.
    local answer

    SPLIT "/" "$options"
    local -a opt_array
    opt_arr=( "${RA[@]}" )

    if [[ "$options" != "" ]]; then
        prompt="$prompt ($options)"
    fi
    if [[ "$default" != "" ]]; then
        prompt="$prompt [$default]"
    fi

    if [[ "$text" != "" ]]; then
        SAY -L -e "$text"
    fi
    while :; do
        read -e -p "$prompt: " answer
        LOG "read: <$answer>."
        if [[ "x$answer" == "x" ]]; then
            answer="$default"
        fi
        if [[ "$options" == "" ]] || IS_ONE_OF "$answer" "${opt_arr[@]}"; then
            SAY -L ""
            RS="$answer"
            LOG "ret: <$answer>"
            return
        fi
        MAKE_LIST "${opt_arr[@]}"
        SAY "Please enter $RS."
    done

} # function ask

function DISPLAY_QUESTION(){
	
	local intro=$1
	local q=$2
	local da=$3
	local rda=$4
	
	IS_COMMAND_LINE_OPTION_EXISTS silent && return ${ERR_OK}
	LOG_ARGS $@

	[ "x$intro" != "x" ] && echo $intro
	echo -n "$q"
	[ ! -z "$da" ] && echo -n " ( $da )"
	[ ! -z "$rda" ] && echo -n " [ $rda ]"
	echo -n ": "
}

function FIND_STRING()
{
    local s=$1
    local a=
    while [ 1 == 1 ]; do
		shift
		[ "$1" == ";" ] && break
    done
	shift
    LOG "Checking '$s' against this list: $@"
    for a in $@
    do
        if [ "x$s" == "x$a" ];then
            LOG "Found"
            return ${ERR_OK}
        fi
    done
    LOG "Haven't found '$s'"
    return 1
}

function COMPARE_VERSION(){
	local A=$1
	local B=$2
        local COMPARE_RESULT=0
        local INDEX=1
        local CA="1"
        local CB="2"

        if [ $(echo $A | grep -v "\.") ] && [ $(echo $B | grep -v "\.") ]; then
		if [ "$A" -gt "$B" ] ; then
		        COMPARE_RESULT=1
		elif [ "$B" -gt "$A" ] ; then
		        COMPARE_RESULT=-1
	        fi # if [ "$A" -gt "$B" ] ; then
		return $COMPARE_RESULT
        fi
       


        while [ "$CA" != "" ] && [ "$CB" != "" ] ; do
		CA=$(echo $A | cut -d'.' -f${INDEX})
		CB=$(echo $B | cut -d'.' -f${INDEX})
		if [ "$CA" != "" ] && [ "$CB" = "" ] ; then
	    		COMPARE_RESULT=1
		elif [ "$CA" = "" ] && [ "$CB" != "" ] ; then
			COMPARE_RESULT=-1
		elif [ "$CA" != "" ] && [ "$CB" != "" ] ; then
	    		if [ "$CA" -gt "$CB" ] ; then
				COMPARE_RESULT=1
			elif [ "$CB" -gt "$CA" ] ; then
				COMPARE_RESULT=-1
	    		fi # if [ "$CA" -gt "$CB" ] ; then
	    		if [ $COMPARE_RESULT -ne 0 ] ; then
				break
	        	fi # if [ "$COMPARE_RESULT" -ne "0" ] ; then
		fi # if [ "$CA" != "" ] && [ "$CB" = "" ] ; then
		INDEX=$(($INDEX+1))
	done #while [ "$CA" != "" ] && [ "$CB" != "" ] ; do
        return $COMPARE_RESULT
	
} # COMPARE_VERSION(){

function JOIN() {


    local joiner="$1"
    shift 1

    JOIN3 "" "" "$joiner" "$@"

} # function join

function JOIN3() {
    local prefix="$1"
    local suffix="$2"
    local joiner="$3"
    shift 3

    local result=""
    if [[ $# -gt 0 ]]; then
        local first; local -a rest
        first="$prefix$1$suffix"; shift 1      # The first item processed separately.
        rest=( "$@" )                          # Copy all the items (but the first).
        rest=( "${rest[@]/#/$joiner$prefix}" ) # Prepend items with joiner and prefix.
        rest=( "${rest[@]/%/$suffix}" )        # Appemd items with suffix.
        local ifs="$IFS"
        IFS=""
        result="$first${rest[*]}"              # And join all the items together.
        IFS="$ifs"
    fi

    RS="$result"

} # function join3

function DIE() {
    local opts=""
    local log="log"
    local prefix="ERROR: "
    while [[ "$1" == -[a-zA-Z] ]]; do
        case "$1" in
            -L ) log="";;
            -P ) prefix="";;
            *  ) opts="$opt $1";;
        esac
        shift 1
    done
    [[ "$log" != "" ]] && LOG "warn:" "$@" "(end)"
    local message
    for message in "$@"; do
        echo $opts "$prefix$message" 1>&2
    done
    BYE 1
} # function die

function BYE() {

    local code="$1"

    [[ -z "$code" ]] && code="0"
    LOG "bye ($code)."

    exit "$code"

} # function bye

function _croak_() {

    local level=$(( $1 + 2 ))
    shift 1

    DIE "Internal error in function \"${FUNCNAME[$level]}\", file \"${BASH_SOURCE[$level]}\", line ${BASH_LINENO[$level-1]}:" \
        "$@" "Please report."

} # function _croak_

function croak0() {

    _croak_ 0 "$@"

} # function croak0

function CROAK() {

    _croak_ 1 "$@"

} # function croak

function INIT_LOG() {

    if [ "x$LOG_F" == "x" ]; then
	local param=$1
	if [ "x$param" == "x" ]; then
	    WARN -L "LOG file was not specified. Logging will be disabled."
	else
	    LOG_F=$param
	fi
    fi
    
    echo "-+ Logging in shell wrapper is started with LOG=$LOG_F +-" 2 > /dev/null >> "$LOG_F"
    if [ $? != 0 ]; then
	WARN -L "Can not write log file \"$LOG_F\". Logging will be disabled."
	unset LOG_F
    fi

} # function init_log

function LOG() {

    if [[ -n "$LOG_F" ]]; then
        local time_stamp=$( date +'%a %b %d %T' )
        local line
        for line in "$@"; do
            { echo "$time_stamp: $line" >> "$LOG_F"; } 2> /dev/null
            if [[ $? -ne 0 ]]; then
                WARN -L "Can not write log file \"$LOG_F\"."
                unset LOG_F  # Unset LOG variable not to log any messages any more.
                break
            fi
        done
    fi

} # function log

function LOG_ARGS() {

    local func="${FUNCNAME[1]}"

    JOIN3 "<" ">" ", " "$@"
    if [[ "$RS" == "" ]]; then
        LOG "$func."
    else
        LOG "$func: $RS."
    fi

} # function log_args

function LOG_VARS() {

    local name str
    local -a vars=()
    for name in "$@"; do
        vars["${#vars[@]}"]="$name=\"${!name}\""
    done
    JOIN ", " "${vars[@]}"
    LOG "vars: $RS."

} # function log_vars

function LOG_FILE() {

    local file="$1"
    local line

    LOG \
        "file <$file>:" \
        "+-------------------------------------------------------------------------------"
    if [[ -f "$file" ]]; then
        while read line; do
            LOG "| $line"
        done < "$file"
    else
        LOG "file not found"
    fi
    LOG "+-------------------------------------------------------------------------------"

} # function log_file

function WARN() {
    local opts=""
    local log="log"
    local prefix="WARNING: "
    while [[ "$1" == -[a-zA-Z] ]]; do
        case "$1" in
            -L ) log="";;
            -P ) prefix="";;
            *  ) opts="$opt $1";;
        esac
        shift 1
    done
    [[ "$log" != "" ]] && LOG_ARGS "$@"
#    local message
#    for message in "$@"; do
#        echo $opts "$prefix$message" 1>&2
#    done
} # function warn

function SAVE_COMMAND_LINE(){
	CMD_STR=$@
	declare -r CMD_STR   
} #SAVE_COMMAND_LINE

function IS_COMMAND_LINE_OPTION_EXISTS(){
                                            
	local cmd=$1                                    
	if echo $CMD_STR | egrep -i "[[:space:]]*\-\-$cmd" &> /dev/null ; then
		return ${ERR_OK}
	fi
	
	return 1
} # is_command_line_option_exists()

function GET_COMMAND_LINE_OPTION_VALUE(){
	
	local cmd=$1
	local err=0;
	
	IS_COMMAND_LINE_OPTION_EXISTS $cmd
	err=$?
	
	[ $err -eq ${ERR_OK} ] || return 1
		
        RS=$(echo $CMD_STR | sed s/.*--$cmd[[:blank:]]*//g | sed 's/[[:blank:]]*--.*$//g')
        [[ -z "$RS" ]] && return 1
        echo $RS
	
} # get_command_line_option_value() {

function PKG_CHECK(){
	local package=$1
	local pack_ver=$2
	local pack_ver_rel=$3
	local pack='';
	local err=$ERR_OK;
	
	local equal=`echo $pack_ver_rel | sed s/[^e]*e[^e]*//g`
	local greater=`echo $pack_ver_rel | sed s/[^g]*g[^g]*//g`
	local less=`echo $pack_ver_rel | sed s/[^l]*l[^l]*//g`

	local seq=1
	local sgt=1
	local slt=1

	if [ "$pack_ver_rel" = ne ] ; then
		pack=`RPM_INFO $package name`
		err=$?
			
		if [ $err -eq ${ERR_RPM_NO_PACKAGE} ]  ; then 
			return ${ERR_OK}; 
		elif [ $err -ne ${ERR_OK} ] ; then
			return $err;
		fi # if [ $err -eq ${ERR_RPM_NO_PACKAGE} ]  ; then
			
		COMPARE_VERSION `RPM_INFO $package version` $pack_ver
			
		if [ $? -ne 0 ] ; then 
			return ${ERR_OK}; 
		fi # if [ $? -ne 0 ] ; then
		
	fi # if [ "$pack_ver_rel" = ne ] ; then

	if [ "$pack_ver_rel" = eq ] || [ "$pack_ver_rel" = ge ] ||[ "$pack_ver_rel" = lt ] ||[ "$pack_ver_rel" = le ] ||[ "$pack_ver_rel" = gt ] ; then 
		pack=`RPM_INFO $package name`; 
		[ $? -eq ${ERR_OK} ] || err=1; 
		if [ $err -eq ${ERR_OK} ] ; then 
			COMPARE_VERSION `RPM_INFO $package version` $pack_ver
			local error=$?

			if test $equal"x" = "x" ; then
				[ $error -eq 0 ] && seq=0 
			fi # if test $equal"x" = "x" ; then
			if test $greater"x" = "x" ; then
				[ $error -eq 1 ] && sgt=0
			fi # if test $greater"x" = "x" ; then
			if [ $less"x" = "x" ] ; then
				[ $error -eq 255 ] && slt=0
			fi # if [ $less"x" = "x" ] ; then
		fi # if [ $err -eq ${ERR_OK} ] ; then
	fi # if [ "$pack_ver_rel" = eq ] || [ "$pack_ver_rel" = ge...
	

	if [ $err -eq ${ERR_OK} ]; then 
		err=1
		case $pack_ver_rel in
			eq ) [ $seq -ne 0 ] || err=0 ;;
			ge ) [ $seq -ne 0 ] || err=0 ; [ $sgt -eq 1 ] || err=0 ;;
			le ) [ $seq -ne 0 ] || err=0 ; [ $slt -eq 1 ] || err=0 ;;
			gt ) [ $seq -eq 1 ] && [ $sgt -eq 0 ] && err=0 ;;
			lt ) [ $seq -eq 1 ] && [ $slt -eq 0 ] && err=0 ;;
		esac # if [ $err -eq ${ERR_OK} ]; then
	fi # if [ $err -eq ${ERR_OK} ]; then

	return $err

} # PKG_CHECK(){

function ARC_GET(){
	arch_tool="uname -m"
	if $($arch_tool | egrep 'i.86' > /dev/null) ; then
		IA='IA32';
		PLATFORM=x32;
	elif $($arch_tool | grep ia64 > /dev/null) ; then 
		IA='IPF'
		PLATFORM=x64
	elif $($arch_tool | grep x86_64 > /dev/null) ; then
		IA='EM64T'
		PLATFORM=x32_64
	else
		LOG "Unknown arch found: $(uname -m)"
		return ${ERR_UNKNOWN_ARC}
	fi # if [ arch | egrep 'i?86' > /dev/null ] ; then
	return ${ERR_OK}
} # ARC_GET(){

declare INTEL_SDP_PRODUCTS_DB=intel_sdp_products.db # TBD...
declare INTEL_SDP_PRODUCTS_DB_LOCAL=''
declare NONRPM_DB_MODE=''
declare NONRPM_DB_PREFIX="$HOME/intel"

function NONRPM_SET_DB_MODE(){
    [ "x$NONRPM_DB_MODE" == "x" ] || return 0
    LOG "NONRPM_SET_DB_MODE"

    INTEL_SDP_PRODUCTS_DB="$NONRPM_DB_PREFIX/$INTEL_SDP_PRODUCTS_DB"
    LOG "INTEL_SDP_PRODUCTS_DB set to $INTEL_SDP_PRODUCTS_DB"

    if IS_COMMAND_LINE_OPTION_EXISTS use-new-db; then
	NONRPM_DB_MODE="both"
    elif IS_COMMAND_LINE_OPTION_EXISTS ignore-old-db; then
	NONRPM_DB_MODE="new"
    else
	NONRPM_DB_MODE="old"
    fi
    LOG "NONRPM_DB_MODE set to $NONRPM_DB_MODE"
    
    if IS_COMMAND_LINE_OPTION_EXISTS nonrpm; then 
	if [ -e "$NONRPM_DB_PREFIX" ]; then
	    if [ ! -d "$NONRPM_DB_PREFIX" ]; then
		LOG "$NONRPM_DB_PREFIX exists and this is not a directory"
		DIE "$NONRPM_DB_PREFIX exists and this is not a directory"
	    fi
	else
	    mkdir -p "$NONRPM_DB_PREFIX" &>/dev/null
	    if [ "$?" != "0" ]; then
		LOG "Unable to create a directory $NONRPM_DB_PREFIX"
		DIE "Unable to create a directory $NONRPM_DB_PREFIX"
	    fi
	fi  
    fi
    
    if [ "$NONRPM_DB_MODE" != "old" ]; then
	LOG "Using INTEL_SDP_PRODUCTS_DB_LOCAL="
	if [ "x$INSTALL_HOST_ID" != "x" ]; then
	    LOG "via INSTALL_HOST_ID var"
	    INTEL_SDP_PRODUCTS_DB_LOCAL="$NONRPM_DB_PREFIX/intel_sdp_products_$INSTALL_HOST_ID.db"
	else
	    LOG "via <hostname>"
	    RS=$(hostname)
	    if [ "$?" != "0" ]; then
		WARN "Unable to define host name, 'only-old-db' mode will be used for Non-rpm functionality."
		INTEL_SDP_PRODUCTS_DB_LOCAL=''
		NONRPM_DB_MODE="old"
	    else
		INTEL_SDP_PRODUCTS_DB_LOCAL="$NONRPM_DB_PREFIX/intel_sdp_products_$RS.db"
	    fi
	fi
	LOG "INTEL_SDP_PRODUCTS_DB_LOCAL set to $INTEL_SDP_PRODUCTS_DB_LOCAL"
	if IS_COMMAND_LINE_OPTION_EXISTS nonrpm; then
	    [ -e "$INTEL_SDP_PRODUCTS_DB_LOCAL" ] || RUN echo -n "" 2>/dev/null 1>$INTEL_SDP_PRODUCTS_DB_LOCAL
	fi
    fi
    
    if IS_COMMAND_LINE_OPTION_EXISTS nonrpm; then
	if [ "$NONRPM_DB_MODE" == "old" ] || [ "$NONRPM_DB_MODE" == "both" ]; then 
	    [ -e "$INTEL_SDP_PRODUCTS_DB" ] || RUN echo -n "" 2>/dev/null 1>$INTEL_SDP_PRODUCTS_DB
	fi
    fi
    
    return 0
}

function NONRPM_GET_BUFFER(){
    
    LOG "NONRPM_GET_BUFFER"
    NONRPM_SET_DB_MODE

    if IS_COMMAND_LINE_OPTION_EXISTS nonrpm; then
	if [ "$NONRPM_DB_MODE" == "old" ] || [ "$NONRPM_DB_MODE" == "both" ]; then 
	    [ -e "$INTEL_SDP_PRODUCTS_DB" ] || RUN echo -n "" 2>/dev/null 1>$INTEL_SDP_PRODUCTS_DB
	fi
	if [ "$NONRPM_DB_MODE" != "old" ]; then
	    [ -e "$INTEL_SDP_PRODUCTS_DB_LOCAL" ] || RUN echo -n "" 2>/dev/null 1>$INTEL_SDP_PRODUCTS_DB_LOCAL
	fi
    fi
    
    local result=''
    local rst=1
    case $NONRPM_DB_MODE in
    new)
	LOG "case new"
	result=$(cat "$INTEL_SDP_PRODUCTS_DB_LOCAL" 2>/dev/null)
	rst=$?
	;;
    both)
	LOG "case both"
	result=$(cat "$INTEL_SDP_PRODUCTS_DB" "$INTEL_SDP_PRODUCTS_DB_LOCAL" 2>/dev/null | sort | uniq)
	rst=$?
	;;
    *)
	LOG "case any"
	result=$(cat "$INTEL_SDP_PRODUCTS_DB" 2>/dev/null)
	rst=$?
	;;
    esac

    if IS_COMMAND_LINE_OPTION_EXISTS nonrpm; then
	[ "$rst" == "0" ] || return 1
    fi
    
    LOG "result:"
    LOG "$result"

    RS=$result

    if [ "$NONRPM_DB_MODE" == "old" ] || [ "$NONRPM_DB_MODE" == "both" ]; then
	dbsize=$(ls -ls "$INTEL_SDP_PRODUCTS_DB" 2>/dev/null | cut -d" " -f1)
	errcode=$?
	if [ "x$errcode" == "x0" ] ; then
	    if [ "x$dbsize" == "x0" ] ; then
		rm -f "$INTEL_SDP_PRODUCTS_DB" &>/dev/null
	    fi
	fi
    fi
    if [ "$NONRPM_DB_MODE" == "new" ] || [ "$NONRPM_DB_MODE" == "both" ]; then
	dbsize=$(ls -ls "$INTEL_SDP_PRODUCTS_DB_LOCAL" 2>/dev/null | cut -d" " -f1)
	errcode=$?
	if [ "x$errcode" == "x0" ] ; then
	    if [ "x$dbsize" == "x0" ] ; then
		rm -f "$INTEL_SDP_PRODUCTS_DB_LOCAL" &>/dev/null
	    fi
	fi
    fi

    return 0
}

function NONRPM_DB_ENTRY_CHECK_SYNTAX() {
    local entry="$1"
    local regexp='<:[^:]*:intel-[a-z_0-9-]+-[0-9]+(\.[0-9]+)+([abpet]u?)?-[0-9]+\.?[a-z_0-9]+\.rpm:[^:]*:[^:]*:>'
    return $(echo "$entry" | grep -E -x "$regexp" > /dev/null) \
        || FATAL_EXIT "Unexpected error."
} # NONRPM_DB_ENTRY_CHECK_SYNTAX

function NONRPM_DB_ENTRY_GET_FIELD() {
    local entry="$1"
    local field="$2"
    RS=''
	if ! NONRPM_DB_ENTRY_CHECK_SYNTAX "$entry"; then
        return 1
    fi
    if [ "$field" -lt 2 ] || [ "$field" -gt 5 ]; then
        return 2
    fi
    local result=$(echo "$entry" | cut -d':' -f"$field") \
        || FATAL_EXIT "Unexpected error"
    RS=$result
    return 0
} # NONRPM_DB_ENTRY_GET_FIELD


function NONRPM_DB_ENTRY_GET_RPMNAME() {
    NONRPM_DB_ENTRY_GET_FIELD "$1" 2
    return $?
} # NONRPM_DB_ENTRY_GET_RPMNAME

function NONRPM_DB_ENTRY_GET_RPMFILE() {
    NONRPM_DB_ENTRY_GET_FIELD "$1" 3
    return $?
} # NONRPM_DB_ENTRY_GET_RPMFILE

function NONRPM_DB_ENTRY_GET_INSTALLDIR() {
    NONRPM_DB_ENTRY_GET_FIELD "$1" 4
    return $?
} # NONRPM_DB_ENTRY_GET_INSTALLDIR

function NONRPM_DB_ENTRY_GET_LOGFILE() {
    NONRPM_DB_ENTRY_GET_FIELD "$1" 5
    return $?
} # NONRPM_DB_ENTRY_GET_LOGFILE

function NONRPM_DB_ENTRY_ADD() {

    LOG "NONRPM_DB_ENTRY_ADD"
    LOG_ARGS $@
    local entry="$1"

    NONRPM_SET_DB_MODE

    if ! NONRPM_DB_ENTRY_CHECK_SYNTAX "$entry"; then
        return 1
    fi

    local db_content=
    if [ "$NONRPM_DB_MODE" == "old" ] || [ "$NONRPM_DB_MODE" == "both" ]; then
	db_content=$(cat "$INTEL_SDP_PRODUCTS_DB" 2>/dev/null)
	echo "$entry" > "$INTEL_SDP_PRODUCTS_DB"
	[ "x$db_content" == "x" ] || echo "$db_content" >> "$INTEL_SDP_PRODUCTS_DB"
	if [ $? -ne 0 ]; then
	    return 2
	fi
    fi

    if [ "$NONRPM_DB_MODE" == "new" ] || [ "$NONRPM_DB_MODE" == "both" ]; then
	db_content=$(cat "$INTEL_SDP_PRODUCTS_DB_LOCAL" 2>/dev/null)
	echo "$entry" > "$INTEL_SDP_PRODUCTS_DB_LOCAL"
	[ "x$db_content" == "x" ] || echo "$db_content" >> "$INTEL_SDP_PRODUCTS_DB_LOCAL"
	if [ $? -ne 0 ]; then
	    return 2
	fi
    fi
    
    return 0
    
} # NONRPM_DB_ENTRY_ADD

function NONRPM_DB_ENTRY_REMOVE() {

    LOG "NONRPM_DB_ENTRY_REMOVE"
    LOG_ARGS $@
    local entry="$1"

    NONRPM_SET_DB_MODE

    if ! NONRPM_DB_ENTRY_CHECK_SYNTAX "$entry"; then
        return 1
    fi

    local dbsize=
    local errcode=
    if [ "$NONRPM_DB_MODE" == "old" ] || [ "$NONRPM_DB_MODE" == "both" ]; then
	cp -p "$INTEL_SDP_PRODUCTS_DB" "$INTEL_SDP_PRODUCTS_DB~" \
	    || DIE "Unable to create backup copy of \"$INTEL_SDP_PRODUCTS_DB\" file."
	grep -F -v -x "$entry" "$INTEL_SDP_PRODUCTS_DB~" > "$INTEL_SDP_PRODUCTS_DB"
	local rc=$?
	[ $rc -le 1 ] || DIE "Unable to overwrite \"$INTEL_SDP_PRODUCTS_DB\" file."
	chmod --reference="$INTEL_SDP_PRODUCTS_DB~" "$INTEL_SDP_PRODUCTS_DB" \
	    || DIE "Unable to change permissions on \"$INTEL_SDP_PRODUCTS_DB\" file."
	rm -f "$INTEL_SDP_PRODUCTS_DB~" &>/dev/null
	dbsize=$(ls -ls "$INTEL_SDP_PRODUCTS_DB" 2>/dev/null | cut -d" " -f1)
	errcode=$?
	if [ "x$errcode" == "x0" ] ; then
	    if [ "x$dbsize" == "x0" ] ; then
		rm -f "$INTEL_SDP_PRODUCTS_DB" &>/dev/null
	    fi
	fi
    fi
    
    if [ "$NONRPM_DB_MODE" == "new" ] || [ "$NONRPM_DB_MODE" == "both" ]; then
	cp -p "$INTEL_SDP_PRODUCTS_DB_LOCAL" "$INTEL_SDP_PRODUCTS_DB_LOCAL~" \
	    || DIE "Unable to create backup copy of \"$INTEL_SDP_PRODUCTS_DB_LOCAL\" file."
	grep -F -v -x "$entry" "$INTEL_SDP_PRODUCTS_DB_LOCAL~" > "$INTEL_SDP_PRODUCTS_DB_LOCAL"
	local rc=$?
	[ $rc -le 1 ] || DIE "Unable to overwrite \"$INTEL_SDP_PRODUCTS_DB_LOCAL\" file."
	chmod --reference="$INTEL_SDP_PRODUCTS_DB_LOCAL~" "$INTEL_SDP_PRODUCTS_DB_LOCAL" \
	    || DIE "Unable to change permissions on \"$INTEL_SDP_PRODUCTS_DB_LOCAL\" file."
	rm -f "$INTEL_SDP_PRODUCTS_DB_LOCAL~" &>/dev/null
	dbsize=$(ls -ls "$INTEL_SDP_PRODUCTS_DB_LOCAL" 2>/dev/null | cut -d" " -f1)
	errcode=$?
	if [ "x$errcode" == "x0" ] ; then
	    if [ "x$dbsize" == "x0" ] ; then
		rm -f "$INTEL_SDP_PRODUCTS_DB_LOCAL" &>/dev/null
	    fi
	fi
    fi
    
    return 0

} # NONRPM_DB_ENTRY_REMOVE

function NONRPM_DB_IS_PACKAGE_INSTALLED() {

    LOG "NONRPM_DB_FIND_FILE_OWNER"
    LOG_ARGS $@
    local package="$1"
    local rc

    NONRPM_GET_BUFFER
    if [ "$?" != "0" ]; then
	DIE "Unable to obtain non-rpm DB content"
    fi

    echo $RS | grep ":$package" &>/dev/null
    rc=$?
    if [ $rc -ge 2 ]; then
        DIE "Unexpected error."
	RS=''
    fi

    RS=$rc
    return $rc
} # NONRPM_DB_IS_PACKAGE_INSTALLED

function NONRPM_DB_FIND_FILE_OWNER() {

    LOG "NONRPM_DB_FIND_FILE_OWNER"
    LOG_ARGS $@
    local file="$1"
    local entry
    local log_file
    local owner

    NONRPM_GET_BUFFER
    if [ "$?" != "0" ]; then
	DIE "Unable to obtain non-rpm DB content"
    fi

    local buffer=$RS
    for entry in $buffer; do
	NONRPM_DB_ENTRY_GET_LOGFILE "$entry"
        log_file=$RS
        if [ $? -eq 0 ] && [ -f "$log_file" ]; then
            owner=$(grep -F -x -l "$file" "$log_file")
            if [ $? -ge 2 ]; then
                DIE "Unexpected error."
            fi
            if [ -n "$owner" ]; then
                RS=$entry
                return 0
            fi
        fi
    done
    
    RS=''
    return 1

} # NONRPM_DB_FIND_FILE_OWNER

function NONRPM_DB_ENTRY_FIND_BY_RPMNAME() {

    LOG "NONRPM_DB_ENTRY_FIND_BY_RPMNAME"
    LOG_ARGS $@
    local rpmname=$(basename "$1")
    local entry
    local r_entries=

    NONRPM_GET_BUFFER
    if [ "$?" != "0" ]; then
	DIE "Unable to obtain non-rpm DB content"
    fi

    local buffer=$RS
    for entry in $buffer; do
	NONRPM_DB_ENTRY_GET_RPMNAME "$entry"
        if [ "$rpmname" == "$RS" ]; then
	    LOG "Entry $entry found"
            RS="$entry"
	    return 0
        fi
    done
    RS=''
    return 1

} # NONRPM_DB_ENTRY_FIND_BY_RPMNAME

function NONRPM_DB_ENTRY_FIND_BY_RPMFILE() {

    LOG "NONRPM_DB_ENTRY_FIND_BY_RPMFILE"
    LOG_ARGS $@
    local rpmfile=$(basename "$1")
    local entry
    local r_entries=

    NONRPM_GET_BUFFER
    if [ "$?" != "0" ]; then
	DIE "Unable to obtain non-rpm DB content"
    fi

    local buffer=$RS
    for entry in $buffer; do
	NONRPM_DB_ENTRY_GET_RPMFILE "$entry"
        if [ "$rpmfile" == "$RS" ]; then
	    LOG "Entry $entry found"
            r_entries="$r_entries $entry"
        fi
    done
    RS=$r_entries
    return 0

} # NONRPM_DB_ENTRY_FIND_BY_RPMFILE

function NONRPM_DB_FIND_BY_INSTALLDIR() {

    LOG "NONRPM_DB_FIND_BY_INSTALLDIR"
    LOG_ARGS $@
    local installdir="$1"
    local entry

    NONRPM_GET_BUFFER
    if [ "$?" != "0" ]; then
	DIE "Unable to obtain non-rpm DB content"
    fi

    local buffer=$RS
    for entry in $buffer; do
	NONRPM_DB_ENTRY_GET_INSTALLDIR "$entry"
        if [ "$installdir" == "$RS" ]; then
	    LOG "Entry $entry found"
            RS=$entry
            return 0
        fi
    done
    
    RS=''
    return 1

} # NONRPM_DB_FIND_BY_INSTALLDIR

function NONRPM_DB_CHECK_SHARED_FILES() {
    local install_dir="$1"
    local log_file="$2"

    local entry
    local line
    local shared=""

    NONRPM_GET_BUFFER
    if [ "$?" != "0" ]; then
	DIE "Unable to obtain non-rpm DB content"
    fi

    local buffer=$RS
    for entry in $buffer; do
	NONRPM_DB_ENTRY_GET_INSTALLDIR "$entry"
	if [ "$install_dir" == "$RS" ]; then
	    NONRPM_DB_ENTRY_GET_LOGFILE "$entry"
	    [ "x$RS" == "x$log_file" ] || shared="$RS $shared"
	fi
    done
    
    local rt
    tac "$log_file" | \
    while read line; do
	rt=1
	if [ "x$shared" != "x" ]; then
	    for i in "$shared"; do
		$(cat $i | grep $line &>/dev/null)
		rt=$?
		[ $rt == 0 ] && break
	    done
	fi
	if [ $rt == 0 ]; then
	    LOG "Shared violation $line"
	    continue
	fi 
        if [[ -h "$line" || -f "$line" ]]; then
            rm -f "$line"
        elif [ -d "$line" ]; then
            rmdir --ignore-fail-on-non-empty "$line"
        else
            echo "Warning: installed \"$line\" file not found" &>$UNINSTALL_LOG_FILE
        fi	
    done
}

function NONRPM_DB_GET_RPMBASED_FIELD(){
	local entry=""$1
	local field=""$2

	LOG "NONRPM_DB_GET_RPMBASED_FIELD:"
	LOG_ARGS $@
	NONRPM_DB_ENTRY_GET_FIELD $entry "2"
	local rpm_name=$RS
	NONRPM_DB_ENTRY_GET_INSTALLDIR "$entry"
	local dir=$RS
	
	LOG "Try to extract $dir/.scripts/$field.$rpm_name"
	if [ -f "$dir/.scripts/$field.$rpm_name" ]; then
		LOG_FILE "$dir/.scripts/$field.$rpm_name"
		RS=$(cat "$dir/.scripts/$field.$rpm_name")
		return 0 
	else
		RS=''
		LOG "$field does not exist in script dir"
	fi
	return 1
}

function CONFIG_CLEAR() {

    local file="$1"

    eval "unset \${!$prefix__*}"  # Unset all the variables starting with prefix.

} # function config_clear

function CONFIG_READ_FILE() {

    LOG_ARGS "$@"
    local clear="yes"

    while [[ "$1" == -[a-zA-Z] ]]; do
        case "$1" in
            -C ) clear="no";;
            *  ) CROAK "$FUNCNAME(): illegal option: $1";;
        esac
        shift 1
    done

    [[ $# -eq 2 ]] || CROAK "$FUNCNAME() expects 2 arguments"

    local prefix="$1"
    local file="$2"

    [[ -e "$file" ]] || DIE "File \"$file\" does not exist."
    [[ -f "$file" ]] || DIE "\"$file\" is not a file."
    [[ -r "$file" ]] || DIE "File \"$file\" is not readable."

    [[ "$clear" == "yes" ]] && CONFIG_CLEAR "$prefix"

    local i=0
    local section
    local name
    local value
    while read line; do
        i=$(( $i + 1 ))
        if [[ "$line" == \#* ]]; then
            : # Skip comments
        elif [[ "$line" == \[*\] ]]; then
            section="$line"
            section="${section#[}"  # Strip opening bracket.
            section="${section%]}"  # Strip closing bracket.
        elif [[ "$line" == *=* ]]; then
            name="${line%%=*}"      # Strip value.
            value="${line#*=}"      # Strip name.
            eval "${prefix}__${section}__${name}=\"\$value\""
        elif [[ "$line" == "" ]]; then
            : # Ignore empty lines.
        else
            DIE "Error in config file \"$file\" at line $i."
        fi
    done < "$file"

} # function config_read_file

function CONFIG_GET_VALUE() {

    local prefix="$1"
    local section="$2"
    local name="$3"

    eval "RS=\"\${${prefix}__${section}__${name}}\""

} # function config_get_value

function CONFIG_SET_VALUE() {

    local prefix="$1"
    local section="$2"
    local name="$3"
    local value="$4"

    eval "${prefix}__${section}__${name}=\"$value\""

} # function config_set_value

function CONFIG_GET_NAMES() {

    local prefix="$1"
    local section="$2"

    eval "RA=( \${!${prefix}__${section}__*} )"
    RA=( "${RA[@]#${prefix}__${section}__}" )   # Strip prefix and section names.

} # function config_get_names

function CONFIG_GET_SECTIONS() {

    local prefix="$1"

    local -a sections

    local -a vars
    eval "vars=( \${!${prefix}__*} )"
    vars=( "${vars[@]#${prefix}__}" )   # Strip prefix.
    vars=( "${vars[@]%%__*}" )          # Strip names.

    local var
    sections=()
    for var in "${vars[@]}"; do
        if ! IS_ONE_OF "$var" "${sections[@]}"; then
	    sections[${#sections[@]}]="$var"
        fi
    done

    RA=( "${sections[@]}" )

} # function config_get_sections

function CONFIG_WRITE_FILE() {

    local prefix="$1"
    local file="$2"

    [[ -e "$file" ]] || $(echo "" 2>/dev/null 1>$file) || DIE "Cannot create duplicate file \"$file\"."
    [[ -f "$file" ]] || DIE "Duplicate path \"$file\" is not a file."
    [[ -w "$file" ]] || DIE "Duplicate file \"$file\" is not writable."

    {
        local -a sections names
        local section name value
        CONFIG_GET_SECTIONS "$prefix"; sections=( "${RA[@]}" )

        for section in "${sections[@]}"; do
            echo "[$section]"
            CONFIG_GET_NAMES "$prefix" "$section"; names=( "${RA[@]}" )
            for name in "${names[@]}"; do
                CONFIG_GET_VALUE "$prefix" "$section" "$name"; value="$RS"
                echo "$name=$value"
            done
            echo ""
        done
    } > "$file"

} # function config_write_file

function RUN() {

    [[ $# -ge 1 ]] || CROAK "$FUNCNAME() expects at least one argument."

    local rc
    LOG_ARGS "$@"
    "$@"
    rc=$?
    LOG "ret: $rc."
    return $rc

} # function run

function MAKE_TEMP_FILE() {

    [[ $# -eq 0 ]] || CROAK "$FUNCNAME() does not expect any arguments."
    LOG_ARGS "$@"
    local temp
    temp=$( RUN mktemp -q "/tmp/install.XXXXXXXX" ) || DIE "Can not create temporary file."
    LOG "ret: <$temp>."
    RS="$temp"

} # create_temp_file

function MAKE_TEMP_DIR() {

    [[ $# -eq 0 ]] || CROAK "$FUNCNAME() does not expect any arguments."
    LOG_ARGS "$@"
    local temp
    temp=$( RUN mktemp -q -d "/tmp/install.XXXXXXXX" ) || DIE "Can not create temporary dir."
    LOG "ret: <$temp>."
    RS="$temp"

} # create_temp_dir

function RPM_CONFIG(){
	
	[ $RPM_CHECK -eq 1 ] || RPM_INIT ; local err=$?	# RPM tool hasn't been checked yet, perform the check
	[ $err -eq ${ERR_OK} ] || return $err

	[ ${RPM_CONFIGURED} -eq -1 ] || return ${RPM_CONFIGURED}

	ARC_GET

	local rpms="4.2.1(x64) 4.1 4.0.2 3.0.5 4.2.2(x64)"
	local rpmi='';

	for rpmi in $rpms ; do
		LOG "Check if RPM supports relocateable packages - $rpmi"
		local ver=`echo $rpmi | sed s/\(.*\)//g`
		local arc=`echo $rpmi | sed s/.*\(//g | sed s/\)//g`
		if [ "$arc" = "$PLATFORM" ] || [ $arc"x" = $ver"x" ] ; then
			PKG_CHECK rpm $ver eq
			if [ $? -eq 0 ] ; then
				LOG "Non-relocatable version of RPM. RPM version: $ver, ARC: $arc"
				PREFIX='';
				RPM_CONFIGURED=${ERR_RPM_NOT_RELOCATABLE}
				return ${ERR_RPM_NOT_RELOCATABLE}
			fi # if [ $? -eq 0 ] ; then
		fi # if [ "$arc" = "$PLATFORM" ] ; then
	done # for rpmi in $rpms ; do
	RPM_CONFIGURED=${ERR_OK};
	return ${ERR_OK}
} # RPM_CONFIG(){

function CHECK_COMMANDS(){
        [ "x$(type -p sed 2>&1)" != "x" ] \
            || DIE "Unable to find 'sed' command, please add its location to your PATH."

        local CHECK_CMD=$(echo "sed egrep cp chmod uniq id rpm rm wc cut uname mkdir rmdir readlink mktemp basename date cpio find cat tac ls gunzip" | sed -e"s/rpm //g")
        local c=''
        for c in ${CHECK_CMD} ; do
	    type -p $c &>/dev/null
	    if [ $? -ne 0 ] ; then
                if [ -f "/etc/mvl-release" ] && [ "$c" == "chkconfig" ] ; then
                : # there is no chkconfig command on MontaVista* CGE Linux*
                else
		    echo "ERROR: unable to find command '$c'."
		    echo "Please add the location to the above commands to your PATH and re-run the script"
		    echo -n "Press Enter to continue."
		    WAIT_ENTER_KEY
		    exit 1
                fi
	    fi
	done
}

ERR_OK=0
ERR_RPM_NO_PACKAGE=10
ERR_UNKNOWN_ARC=11
ERR_RPM_UNINSTALL=12
ERR_RPM_NOT_RELOCATABLE=13
ERR_RPM_LOCK=14
declare RS
declare -a RA
declare _USER_ID_=
PLATFORM=''
IA=''
CMD_STR=''
declare LOG_F=
declare -r TELL_WIDTH=80
declare _SET_FMT_=
RPM_CONFIGURED=-1
RPM_CHECK=0

function USER_ID() {

    if [[ "$_USER_ID_" == "" ]]; then
        _USER_ID_=$( id -u ) || DIE "\"id\" failed."
        LOG "User id is \"$_USER_ID_\"."
    fi
    RS="$_USER_ID_"

} # function user_id

function CHECK_ROOT_PRIVILEDGES() {

        USER_ID
        if [ $_USER_ID_ -ne 0 ] ; then
            if [ ! -w /dev ] ; then
                local current_dir=`pwd`
                WARN "Super-user or \"root\" privileges are required in order to continue."
		IS_COMMAND_LINE_OPTION_EXISTS silent && return 1
                echo -n "Please enter \"root\" "
                exec su -c "cd '$current_dir' && /bin/sh '${SCRIPT}' $@" # always do the fork.
                echo "" # should never get here
                return 1
            fi
        fi

}

function UNINSTALL_RPM(){

	local ChosenRPM=$1
	shift
	local uninstall_options=$@
	local fcode=$ERR_OK

	RPM_CONFIG
	    
	local pack_ver=`RPM_INFO $ChosenRPM VERSION`
	fcode=$?
    
	[ $fcode -eq ${ERR_OK} ] || return $fcode
    
	MAKE_TEMP_FILE
    	local TEMPFILE=$RS
	local err=$?
	[ $err -eq ${ERR_OK} ] || return $err
	
	rpm -e --nodeps --allmatches $ChosenRPM 2>&1 &> $TEMPFILE
	local err=$?
	
	if [ $err -ne 0 ] ; then 
		local RPM_LOGS=$(sed 's/^/    /g' $TEMPFILE)
		LOG "Uninstallation of the $ChosenRPM. RPM logs: \n$RPM_LOGS"
		fcode=$ERR_RPM_UNINSTALL
	else
		LOG "Uninstallation of the $ChosenRPM has succeeded."
		fcode=${ERR_OK}
	fi # if [ ! "$RPMERROR" = 0 ] ; then 

	rm -f $TEMPFILE
	RS=$fcode
	return $fcode

} #UNINSTALL_RPM(){

function NONRPM_UNINSTALL_PACKAGE() {

    local entry="$1"
    local noscripts=$2

    if ! NONRPM_DB_ENTRY_CHECK_SYNTAX "$entry"; then
        echo "Specified db entry to uninstall does not look like a valid one. Probably, internal error."
        return 1
    fi

    NONRPM_DB_ENTRY_GET_LOGFILE "$entry"
    local log_file=$RS
    if [ ! -f "$log_file" ]; then
	    echo "Uninstallation cannot continue for this component: Missing \"$log_file\"."
	    return 1
    fi

    NONRPM_DB_ENTRY_GET_INSTALLDIR "$entry"
    local install_dir=$RS
    if [ ! -d "$install_dir" ]; then
	   echo "Uninstallation cannot continue for this component: Missing \"$install_dir\" directory."
	   return 1
    fi

    local script_dir="$install_dir/.scripts"
    if [ "$noscripts" != "1" ]; then
        NONRPM_DB_ENTRY_GET_FIELD "$entry" 2
        local rpm_name=$RS
        NONRPM_DB_ENTRY_GET_RPMFILE "$entry" 
        local rpm_arch=$RS
	rpm_arch=$(echo "$rpm_arch" | sed -e"s/\.rpm//g" -e"s/.*\.//g")
	[ "x$rpm_arch" == "x" ] && rpm_arch="noarch"

        if [[ -f "$script_dir/PREUN.$rpm_name.$rpm_arch" ]]; then
            env RPM_INSTALL_PREFIX="$install_dir" /bin/bash "$script_dir/PREUN.$rpm_name.$rpm_arch" &>/dev/null
        fi
    fi

    NONRPM_DB_CHECK_SHARED_FILES "$install_dir" "$log_file"
    
    if [ "$noscripts" != "1" ]; then
        if [[ -f "$script_dir/POSTUN.$rpm_name.$rpm_arch" ]]; then
            env RPM_INSTALL_PREFIX="$install_dir" /bin/bash "$script_dir/POSTUN.$rpm_name.$rpm_arch" &>/dev/null
        fi
    fi

    local script
    for script in PREIN POSTIN PREUN POSTUN SUMMARY; do
        rm -f "$script_dir/$script.$rpm_name.$rpm_arch"
    done

    if [ "x$(ls $script_dir 2>/dev/null)" == "x" ]; then
	LOG "Nonrpm script dir is emplty. Will be deleted."
	rm -rf "$script_dir"
    else
	LOG "Nonrpm script dir is not emplty."
    fi

    if [[ -d "$install_dir" ]]; then
        rmdir --ignore-fail-on-non-empty "$install_dir"
    fi

    rm -f "$log_file"
    NONRPM_DB_ENTRY_REMOVE "$entry"

    RS=0
    return 0

} # NONRPM_UNINSTALL_PACKAGE

function FATAL_EXIT() {
    DIE $@
}

function READ_RPM_DB() {
    [ $GUID == 0 ] || return 1
    list=$(rpm -qa 2>/dev/null | grep intel | grep ${MAIN_PACKAGE_NUMBER})
    for one in $list ; do
	archin_mode=
	if [ "x$(echo ${one} | grep noarch)" != "x" ] ; then
	    one=$(echo ${one} | sed -e"s/\.noarch$/-noarch/g")
	    archin_mode=1
	elif [ "x$(echo ${one} | grep i486)" != "x" ] ; then
	    one=$(echo ${one} | sed -e"s/\.i486$/-i486/g")
	    archin_mode=1
	elif [ "x$(echo ${one} | grep x86_64)" != "x" ] ; then
	    one=$(echo ${one} | sed -e"s/\.x86_64$/-x86_64/g")
	    archin_mode=1
	else
	    archin_mode=
	fi
	if [ "x${archin_mode}" == "x1" ] ; then
	    RPMDB="${RPMDB} ${one}"
	else
	    llarch=$(rpm -q --qf %{ARCH} ${one} 2>/dev/null)
	    result=$?
	    if [ "$result" == "0" ] ; then
		if [ "$llarch" == "i486x86_64" ] || [ "$llarch" == "x86_64i486" ] ; then
		    RPMDB="${RPMDB} ${one}-i486 ${one}-x86_64"
		elif [ "$llarch" == "i486" ] ; then
		    RPMDB="${RPMDB} ${one}-i486"
		elif [ "$llarch" == "x86_64" ] ; then
		    RPMDB="${RPMDB} ${one}-x86_64"
		else
		    RPMDB="${RPMDB} ${one}-noarch"    
		fi
	    fi
	fi
    done
}

function IS_RPM_INSTALLED()
{
    local package="$1"
    local product
    local rc=1

    [ $GUID == 0 ] || return 1

    product=$(echo $package | sed -e"s/\.rpm$//g" -e"s/\.[a-z0-9_]*$//g")
    localarch=$(echo $package | sed -e"s/\.rpm$//g" -e"s/^.*\.\([a-z0-9_]*\)$/\1/g")
#"
    if [ "x${RPMDB}" == "x" ] ; then 
	rpm -q "$product" &>/dev/null
	rc=$?
	if [ "$rc" == "0" ] ; then
	    if [ "x$(rpm -q --qf %{ARCH} $product 2>/dev/null | grep $localarch)" == "x" ] ; then
    		rc=1
	    else
		rc=0
	    fi
	fi
    else
	result=$(echo "${RPMDB}" | grep ${product}-${localarch})
	if [ "x${result}" == "x" ] ; then
	    rc=1
	else
	    rc=0
	fi
    fi
    
    return $rc
}

function IS_NONRPM_INSTALLED()
{
    local package="$1"
    local product
    local rc=1

    if [ "x$rc" != "x0" ] ; then
        NONRPM_DB_IS_PACKAGE_INSTALLED "$package"
	rc=$?	    
    fi
    return $rc
}

function REMOVE_EMPTY_DIRS()
{
    local rootnode="$1"
    local remove_child_only="$2"
 
    if [ -z "$rootnode" ]; then
       return
    fi

    for dir in $(ls -A $rootnode 2>/dev/null); do
        [[ -d "$rootnode/$dir" ]] && REMOVE_EMPTY_DIRS "$rootnode/$dir" "1"
    done

    rmdir --ignore-fail-on-non-empty $rootnode 1>/dev/null 2>&1
    if [ -z "$remove_child_only" ]; then
        rootnode=$(dirname $rootnode 2> /dev/null)
        while test -z $(echo $rootnode | egrep '^(/|/bin|/boot|/dev|/etc|/lib|/media|/mnt|/opt|/sbin|/srv|/tmp|/usr|/var)$') ; do
           rmdir $rootnode 1>/dev/null 2>&1
           rootnode=$(dirname $rootnode 2> /dev/null)
        done
    fi
}

function print_help()
{
    echo "Usage: uninstall.sh [--silent | --help]"
    echo "		--silent       invoke non interactive uninstall process"
    echo "		--help         print this message and exit"
    echo ""
    echo "Copyright 2006-2013, Intel Corporation, All Rights Reserved."
}

declare RPM_PACKAGES=

declare CONFIG_PREFIX='SILENT'
declare CONFIG_SECTION='CPRO_UNINSTALL'

declare IC_IA32_RPM_PACKAGES="intel-openmp-146-13.1-2.i486.rpm
intel-openmp-devel-146-13.1-2.i486.rpm
intel-compilerpro-common-146-13.1-2.noarch.rpm
intel-compilerpro-vars-146-13.1-2.noarch.rpm
intel-compilerpro-devel-146-13.1-2.i486.rpm
intel-compilerproc-common-146-13.1-2.noarch.rpm
intel-compilerproc-devel-146-13.1-2.i486.rpm
intel-sourcechecker-common-146-13.1-2.noarch.rpm
intel-sourcechecker-devel-146-13.1-2.i486.rpm
intel-compilerproc-146-13.1-2.i486.rpm
"
declare IC_IA32_RPM_SIGNATURE="intel-compilerproc-devel-146-13.1-2.i486.rpm"

declare IC_INTEL64_RPM_PACKAGES="intel-openmp-146-13.1-2.x86_64.rpm
intel-openmp-devel-146-13.1-2.x86_64.rpm
intel-compilerpro-common-146-13.1-2.noarch.rpm
intel-compilerpro-vars-146-13.1-2.noarch.rpm
intel-compilerpro-devel-146-13.1-2.x86_64.rpm
intel-compilerproc-common-146-13.1-2.noarch.rpm
intel-compilerproc-devel-146-13.1-2.x86_64.rpm
intel-sourcechecker-common-146-13.1-2.noarch.rpm
intel-sourcechecker-devel-146-13.1-2.x86_64.rpm
intel-compilerproc-146-13.1-2.x86_64.rpm
"
declare IC_INTEL64_RPM_SIGNATURE="intel-compilerproc-devel-146-13.1-2.x86_64.rpm"

declare IF_IA32_RPM_PACKAGES="intel-openmp-146-13.1-2.i486.rpm
intel-openmp-devel-146-13.1-2.i486.rpm
intel-compilerpro-common-146-13.1-2.noarch.rpm
intel-compilerpro-vars-146-13.1-2.noarch.rpm
intel-compilerpro-devel-146-13.1-2.i486.rpm
intel-compilerprof-common-146-13.1-2.noarch.rpm
intel-compilerprof-146-13.1-2.i486.rpm
intel-compilerprof-devel-146-13.1-2.i486.rpm
intel-sourcechecker-common-146-13.1-2.noarch.rpm
intel-sourcechecker-devel-146-13.1-2.i486.rpm
"
declare IF_IA32_RPM_SIGNATURE="intel-compilerprof-devel-146-13.1-2.i486.rpm"

declare IF_INTEL64_RPM_PACKAGES="intel-openmp-146-13.1-2.x86_64.rpm
intel-openmp-devel-146-13.1-2.x86_64.rpm
intel-compilerpro-common-146-13.1-2.noarch.rpm
intel-compilerpro-vars-146-13.1-2.noarch.rpm
intel-compilerpro-devel-146-13.1-2.x86_64.rpm
intel-compilerprof-common-146-13.1-2.noarch.rpm
intel-compilerprof-146-13.1-2.x86_64.rpm
intel-compilerprof-devel-146-13.1-2.x86_64.rpm
intel-sourcechecker-common-146-13.1-2.noarch.rpm
intel-sourcechecker-devel-146-13.1-2.x86_64.rpm
"
declare IF_INTEL64_RPM_SIGNATURE="intel-compilerprof-devel-146-13.1-2.x86_64.rpm"

declare IDB_IA32_RPM_PACKAGES="intel-compilerpro-common-146-13.1-2.noarch.rpm
intel-idb-common-146-13.0-2.noarch.rpm
intel-idbcdt-146-13.0-2.noarch.rpm
intel-idb-146-13.0-2.i486.rpm
"
declare IDB_IA32_RPM_SIGNATURE="intel-idb-146-13.0-2.i486.rpm"

declare IDB_INTEL64_RPM_PACKAGES="intel-compilerpro-common-146-13.1-2.noarch.rpm
intel-idb-common-146-13.0-2.noarch.rpm
intel-idbcdt-146-13.0-2.noarch.rpm
intel-idb-146-13.0-2.x86_64.rpm
"
declare IDB_INTEL64_RPM_SIGNATURE="intel-idb-146-13.0-2.x86_64.rpm"

declare MKL_IA32_RPM_PACKAGES="intel-openmp-146-13.1-2.i486.rpm
intel-openmp-devel-146-13.1-2.i486.rpm
intel-compilerpro-common-146-13.1-2.noarch.rpm
intel-mkl-common-146-11.0-2.noarch.rpm
intel-mkl-146-11.0-2.i486.rpm
intel-mkl-devel-146-11.0-2.i486.rpm
"
declare MKL_IA32_RPM_SIGNATURE="intel-mkl-devel-146-11.0-2.i486.rpm"

declare MKL_INTEL64_RPM_PACKAGES="intel-openmp-146-13.1-2.x86_64.rpm
intel-openmp-devel-146-13.1-2.x86_64.rpm
intel-compilerpro-common-146-13.1-2.noarch.rpm
intel-mkl-common-146-11.0-2.noarch.rpm
intel-mkl-146-11.0-2.x86_64.rpm
intel-mkl-devel-146-11.0-2.x86_64.rpm
"
declare MKL_INTEL64_RPM_SIGNATURE="intel-mkl-devel-146-11.0-2.x86_64.rpm"

declare TBB_RPM_PACKAGES="intel-compilerpro-common-146-13.1-2.noarch.rpm
intel-tbb-146-4.1-2.noarch.rpm
intel-tbb-devel-146-4.1-2.noarch.rpm
intel-tbb-source-146-4.1-2.noarch.rpm
"
declare TBB_RPM_SIGNATURE="intel-tbb-devel-146-4.1-2.noarch.rpm"

declare IPP_IA32_RPM_PACKAGES="intel-openmp-146-13.1-2.i486.rpm
intel-openmp-devel-146-13.1-2.i486.rpm
intel-compilerpro-common-146-13.1-2.noarch.rpm
intel-ipp-common-146-7.1-1.noarch.rpm
intel-ipp-146-7.1-1.i486.rpm
intel-ipp-devel-146-7.1-1.i486.rpm
"
declare IPP_IA32_RPM_SIGNATURE="intel-ipp-devel-146-7.1-1.i486.rpm"

declare IPP_LP32_RPM_PACKAGES="intel-openmp-146-13.1-2.i486.rpm
intel-openmp-devel-146-13.1-2.i486.rpm
intel-compilerpro-common-146-13.1-2.noarch.rpm
intel-ipp-common-146-7.1-1.noarch.rpm
intel-ipp-146-7.1-1.i686.rpm
intel-ipp-devel-146-7.1-1.i686.rpm
"
declare IPP_LP32_RPM_SIGNATURE="intel-ipp-devel-146-7.1-1.i686.rpm"

declare IPP_INTEL64_RPM_PACKAGES="intel-openmp-146-13.1-2.x86_64.rpm
intel-openmp-devel-146-13.1-2.x86_64.rpm
intel-compilerpro-common-146-13.1-2.noarch.rpm
intel-ipp-common-146-7.1-1.noarch.rpm
intel-ipp-146-7.1-1.x86_64.rpm
intel-ipp-devel-146-7.1-1.x86_64.rpm
"
declare IPP_INTEL64_RPM_SIGNATURE="intel-ipp-devel-146-7.1-1.x86_64.rpm"

declare CRYPTO_IPP_IA32_RPM_PACKAGES="intel-crypto-ipp-common-146-7.1-1.noarch.rpm 
intel-crypto-ipp-146-7.1-1.i486.rpm 
intel-crypto-ipp-devel-146-7.1-1.i486.rpm"
declare CRYPTO_IPP_IA32_RPM_SIGNATURE="intel-crypto-ipp-devel-146-7.1-1.i486.rpm"

declare CRYPTO_IPP_INTEL64_RPM_PACKAGES="intel-crypto-ipp-common-146-7.1-1.noarch.rpm 
intel-crypto-ipp-146-7.1-1.x86_64.rpm 
intel-crypto-ipp-devel-146-7.1-1.x86_64.rpm"
declare CRYPTO_IPP_INTEL64_RPM_SIGNATURE="intel-crypto-ipp-devel-146-7.1-1.x86_64.rpm"

declare CRYPTO_IPP_LP32_RPM_PACKAGES="intel-crypto-ipp-common-146-7.1-1.noarch.rpm 
intel-crypto-ipp-146-7.1-1.i686.rpm 
intel-crypto-ipp-devel-146-7.1-1.i686.rpm"
declare CRYPTO_IPP_LP32_RPM_SIGNATURE="intel-crypto-ipp-devel-146-7.1-1.i686.rpm"

declare GEN_IPP_IA32_RPM_PACKAGES="intel-gen-ipp-common-146-7.1-1.noarch.rpm 
intel-gen-ipp-146-7.1-1.i486.rpm 
intel-gen-ipp-devel-146-7.1-1.i486.rpm"
declare GEN_IPP_IA32_RPM_SIGNATURE="intel-gen-ipp-devel-146-7.1-1.i486.rpm"

declare GEN_IPP_INTEL64_RPM_PACKAGES="intel-gen-ipp-common-146-7.1-1.noarch.rpm 
intel-gen-ipp-146-7.1-1.x86_64.rpm 
intel-gen-ipp-devel-146-7.1-1.x86_64.rpm"
declare GEN_IPP_INTEL64_RPM_SIGNATURE="intel-gen-ipp-devel-146-7.1-1.x86_64.rpm"

declare THREADED_IPP_IA32_RPM_PACKAGES="intel-threaded-static-ipp-devel-146-7.1-1.i486.rpm"
declare THREADED_IPP_IA32_RPM_SIGNATURE="intel-threaded-static-ipp-devel-146-7.1-1.i486.rpm"

declare THREADED_IPP_INTEL64_RPM_PACKAGES="intel-threaded-static-ipp-devel-146-7.1-1.x86_64.rpm"
declare THREADED_IPP_INTEL64_RPM_SIGNATURE="intel-threaded-static-ipp-devel-146-7.1-1.x86_64.rpm"

declare MAIN_PACKAGE_NUMBER="146"

SCRIPT="$0"
CHECK_COMMANDS
SAVE_COMMAND_LINE $@

if IS_COMMAND_LINE_OPTION_EXISTS help ; then
    print_help
    exit 0
fi

# localization
EXECDIR=$(dirname "$SCRIPT")
[ -L "$EXECDIR" ] && EXECDIR=$(readlink $EXECDIR &>/dev/null)
REL_TO_ABS "$EXECDIR"
rc=$?
[ "x$rc" == "x0" ] && EXECDIR="$RS"

if [ "$LANG" != "" ]; then
    case $LANG in
        ja_JP.utf-8 | ja_JP.UTF-8 | ja_JP.utf8 | ja_JP.UTF8)
            LANG_FILE="$EXECDIR/ja_JP/uninstall.txt"
	    ;;
        en_US.utf-8 | en_US.UTF-8)
            LANG_FILE="$EXECDIR/en_US/uninstall.txt"
	    ;;
        * )
            LANG_FILE="$EXECDIR/en_US/uninstall.txt"
	    ;;
    esac
else
    LANG_FILE="$EXECDIR/en_US/uninstall.txt"
fi

if [ ! -e "$LANG_FILE" ] ; then # if unreachible target l12n to English
    LANG_FILE="$EXECDIR/en_US/uninstall.txt"
fi

# if this file is missing something is broken
if [ -s "$LANG_FILE" ]; then
#    echo "Localization file picked up -> $LANG_FILE"
    source $LANG_FILE
else
    echo "Internal error: cannot find localization component"
    rm -f /tmp/${GUID}.composerxe2013.install.semaphore &>/dev/null
    exit 0
fi

declare COMPONENT_LIST="IC IF IDB MKL TBB IPP CRYPTO_IPP GEN_IPP THREADED_IPP"

declare SUMMARY="${UNS_STRING_1}"

uid=$(id -u)
if [ $? != 0 ]; then
    WARN "${UNS_STRING_2}"
    uid=1
fi
GUID=$uid

# semaphore the uninstall instance has been runned for further detection
:>/tmp/${GUID}.composerxe2013.install.semaphore &>/dev/null
chmod 444 /tmp/${GUID}.composerxe2013.install.semaphore &>/dev/null
    
if [ $uid == 0 ]; then
    NONRPM_DB_PREFIX=/opt/intel
    UNINSTALL_LOG_FILE="/var/log/intel-composerxe-146-13.1-2_uninstall.log"
else
    NONRPM_DB_PREFIX="$HOME/intel"
    UNINSTALL_LOG_FILE="$HOME/intel-composerxe-146-13.1-2_uninstall.log"
fi
NONRPM_SET_DB_MODE

SHARED_LOCATION=$(echo "$EXECDIR" | grep -e"\/composer_xe_2013\.[0-9]\.[0-9][0-9][0-9]\/bin" | sed -e"s/\/composer_xe_2013\.[0-9]\.[0-9][0-9][0-9]\/bin//g")

CURRDIR="$(pwd)"
postfix="$EXECDIR"
[ "x$postfix" == "x." ] && postfix=""
if [ "x$(echo "$postfix" | egrep ^/)" != "x" ]; then
    CURRDIR=$postfix
    postfix=""
fi

rs=$(pushd $CURRDIR/$postfix 2>/dev/null)
[ $? != 0 ] && RMINSTALLDIR=""
for i in $rs; do    
    RMINSTALLDIR=$i
    break
done
if [ "x$(echo "$RMINSTALLDIR" | egrep ^~)" != "x" ]; then
    RMINSTALLDIR="$HOME$(echo "$RMINSTALLDIR" | sed -e"s/^~//g")"
fi
RMUPPERDIR=$(dirname "$RMINSTALLDIR")

REL_TO_ABS "$RMUPPERDIR"
rc=$?
[ "x$rc" == "x0" ] && RMUPPERDIR="$RS"

DIR_BELONG_TO_RPM=1
if [ RPM_INIT ]; then
   rpm -qf $RMUPPERDIR &>/dev/null
   DIR_BELONG_TO_RPM=$?
fi

NONRPM_DB_FIND_BY_INSTALLDIR "$RMUPPERDIR"
if ! [[ -n "$RS" ]] && ! [[ $uid == 0 && $DIR_BELONG_TO_RPM == 0 ]] ; then
   echo "${UNS_STRING_26}"
   WAIT_ENTER_KEY
   rm -f /tmp/${GUID}.composerxe2013.install.semaphore &>/dev/null
   exit 1
fi

SCRIPT_ARGS=""
while [ $# -ne 0 ]
do
    SCRIPT_ARGS="$SCRIPT_ARGS \"$1\""
    shift
done
INIT_LOG "$UNINSTALL_LOG_FILE"

RPMDB=
READ_RPM_DB

declare COMPONENTS_INSTALLED=
declare COMPONENTS_TO_UNINSTALL=
declare COMPONENTS_TO_KEEP=
declare RPMS_TO_UNINSTALL=

IS_COMMAND_LINE_OPTION_EXISTS silent || echo -n "${UNS_STRING_3} .."

for one in $COMPONENT_LIST ; do
    IS_COMMAND_LINE_OPTION_EXISTS silent || echo -n "."
    eval "${one}_SELECTED="
    eval "${one}_INDEX="
    if [ "x${one}" == "xTBB" ] ; then
	eval "value=\${${one}_RPM_SIGNATURE}"
	eval "${one}_INSTALLED="
	if IS_RPM_INSTALLED "${value}" ; then
	    LOG "Core ${value} file detected in RPM mode ..."
	    eval "${one}_INSTALLED=1"
	else
	    if IS_NONRPM_INSTALLED "${value}" ; then
		LOG "Core ${value} file detected in NONRPM mode ..."
		eval "${one}_INSTALLED=1"
	    fi
	fi
    else
	eval "value=\${${one}_IA32_RPM_SIGNATURE}"
	eval "${one}_IA32_INSTALLED="
	if IS_RPM_INSTALLED "${value}" ; then
	    LOG "Core ${value} file detected in RPM mode ..."
	    eval "${one}_IA32_INSTALLED=1"
	else
	    if IS_NONRPM_INSTALLED "${value}" ; then
		LOG "Core ${value} file detected in NONRPM mode ..."
		eval "${one}_IA32_INSTALLED=1"
	    fi
	fi
	eval "value=\${${one}_INTEL64_RPM_SIGNATURE}"
	eval "${one}_INTEL64_INSTALLED="
	if IS_RPM_INSTALLED "${value}" ; then
	    LOG "Core ${value} file detected in RPM mode ..."
	    eval "${one}_INTEL64_INSTALLED=1"
	else
	    if IS_NONRPM_INSTALLED "${value}" ; then
		LOG "Core ${value} file detected in NONRPM mode ..."
		eval "${one}_INTEL64_INSTALLED=1"
	    fi
	fi
	if [ "x$one" == "xIPP" ] || [ "x$one" == "xCRYPTO_IPP" ] ; then
	    eval "value=\${${one}_LP32_RPM_SIGNATURE}"
	    eval "${one}_LP32_INSTALLED="
	    if IS_RPM_INSTALLED "${value}" ; then
		LOG "Core ${value} file detected in RPM mode ..."
		eval "${one}_LP32_INSTALLED=1"
	    else
		if IS_NONRPM_INSTALLED "${value}" ; then
		    LOG "Core ${value} file detected in NONRPM mode ..."
		    eval "${one}_LP32_INSTALLED=1"
		fi
	    fi
	fi
    fi
done

any_installed=
for one in $COMPONENT_LIST ; do
    IS_COMMAND_LINE_OPTION_EXISTS silent || echo -n "."
    if [ "x${one}" == "xTBB" ] ; then
	eval "value=\${${one}_INSTALLED}"
	[ "x$value" == "x1" ] && any_installed=1 && COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED ${one}"
    else
	eval "value=\${${one}_IA32_INSTALLED}"
	[ "x$value" == "x1" ] && any_installed=1 && COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED ${one}_IA32"
	eval "value=\${${one}_INTEL64_INSTALLED}"
	[ "x$value" == "x1" ] && any_installed=1 && COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED ${one}_INTEL64"

	if [ "x$one" == "xIPP" ] || [ "x$one" == "xCRYPTO_IPP" ] ; then
	    eval "value=\${${one}_LP32_INSTALLED}"
	    [ "x$value" == "x1" ] && any_installed=1 && COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED ${one}_LP32"
	fi
    fi
done

if [ "x${any_installed}" != "x1" ] ; then
    echo
    echo "${UNS_STRING_4} $SUMMARY:"
    echo "${UNS_STRING_5}"
    echo "${UNS_STRING_6} \"$UNINSTALL_LOG_FILE\" ${UNS_STRING_7}"
    if ! IS_COMMAND_LINE_OPTION_EXISTS silent; then
        echo "${UNS_STRING_20}"
	WAIT_ENTER_KEY
    fi
    rm -f /tmp/${GUID}.composerxe2013.install.semaphore &>/dev/null
    exit 0
fi

IS_COMMAND_LINE_OPTION_EXISTS silent || echo

ACCEPT_UNINSTALL=1
while [ $ACCEPT_UNINSTALL -eq 1 ] ; do

COMPONENTS_TO_UNINSTALL=
COMPONENTS_TO_KEEP=
RPMS_TO_UNINSTALL=

if ! IS_COMMAND_LINE_OPTION_EXISTS silent ; then
exit_flag_0=
select_all=0
while [ 1 -eq 1 ] ; do
    [ "x$exit_flag_0" == "x1" ] &> /dev/null && break
    clear
    INDEX=0
    COMPONENTS_INSTALLED=
    echo "${UNS_STRING_8} $SUMMARY."
    echo ""
    echo "${UNS_STRING_9}"
    echo ""
    echo "     ${INDEX}. ${UNS_STRING_10}"
    if [ "$IC_IA32_INSTALLED" == "1" ] || [ "$IC_INTEL64_INSTALLED" == "1" ] ; then
	INDEX=$(($INDEX+1))
	IC_INDEX=$INDEX
	IC_INSTALLED=1
	line2=
	if [ "$IC_IA32_INSTALLED" == "1" ] && [ "$IC_INTEL64_INSTALLED" == "1" ] ; then
	    COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IC_IA32 IC_INTEL64"
	    line="${UNS_PRODUCT_1}"
	    line2="            ${UNS_STRING_21} ${UNS_PRODUCT_2}"
	else
	    if [ "$IC_IA32_INSTALLED" == "1" ] ; then
		COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IC_IA32"
		line="${UNS_PRODUCT_1}"
	    fi
	    if [ "$IC_INTEL64_INSTALLED" == "1" ] ; then
		COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IC_INTEL64"
		line="${UNS_PRODUCT_2}"
	    fi
	fi
	if [ "$IC_SELECTED" == "1" ] ; then
	    line=" [x] ${INDEX}. $line"
	else
	    line=" [ ] ${INDEX}. $line"
	fi
	echo "$line"
	[ "x$line2" == "x" ] || echo "$line2"
    fi

    if [ "$IF_IA32_INSTALLED" == "1" ] || [ "$IF_INTEL64_INSTALLED" == "1" ] ; then
	INDEX=$(($INDEX+1))
	IF_INDEX=$INDEX
	IF_INSTALLED=1
	line2=
	if [ "$IF_IA32_INSTALLED" == "1" ] && [ "$IF_INTEL64_INSTALLED" == "1" ] ; then
	    COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IF_IA32 IF_INTEL64"
	    line="${UNS_PRODUCT_3}"
	    line2="            ${UNS_STRING_21} ${UNS_PRODUCT_4}"
	else
	    if [ "$IF_IA32_INSTALLED" == "1" ] ; then
		COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IF_IA32"
		line="${UNS_PRODUCT_3}"
	    fi
	    if [ "$IF_INTEL64_INSTALLED" == "1" ] ; then
		COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IF_INTEL64"
		line="${UNS_PRODUCT_4}"
	    fi
	fi
	if [ "$IF_SELECTED" == "1" ] ; then
	    line=" [x] ${INDEX}. $line"
	else
	    line=" [ ] ${INDEX}. $line"
	fi
	echo "$line"
	[ "x$line2" == "x" ] || echo "$line2"
    fi

    if [ "$IDB_IA32_INSTALLED" == "1" ] || [ "$IDB_INTEL64_INSTALLED" == "1" ] ; then
	INDEX=$(($INDEX+1))
	IDB_INDEX=$INDEX
	IDB_INSTALLED=1
	line2=
	if [ "$IDB_IA32_INSTALLED" == "1" ] && [ "$IDB_INTEL64_INSTALLED" == "1" ] ; then
	    COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IDB_IA32 IDB_INTEL64"
	    line="${UNS_PRODUCT_5}"
	    line2="            ${UNS_STRING_21} ${UNS_PRODUCT_6}"
	else
	    if [ "$IDB_IA32_INSTALLED" == "1" ] ; then
		COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IDB_IA32"
		line="${UNS_PRODUCT_5}"
	    fi
	    if [ "$IDB_INTEL64_INSTALLED" == "1" ] ; then
		COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IDB_INTEL64"
		line="${UNS_PRODUCT_6}"
	    fi
	fi
	if [ "$IDB_SELECTED" == "1" ] ; then
	    line=" [x] ${INDEX}. $line"
	else
	    line=" [ ] ${INDEX}. $line"
	fi
	echo "$line"
	[ "x$line2" == "x" ] || echo "$line2"
    fi

    if [ "$MKL_IA32_INSTALLED" == "1" ] || [ "$MKL_INTEL64_INSTALLED" == "1" ] ; then
	INDEX=$(($INDEX+1))
	MKL_INDEX=$INDEX
	MKL_INSTALLED=1
	line2=
	if [ "$MKL_IA32_INSTALLED" == "1" ] && [ "$MKL_INTEL64_INSTALLED" == "1" ] ; then
	    COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED MKL_IA32 MKL_INTEL64"
	    line="${UNS_PRODUCT_7}"
	    line2="            ${UNS_STRING_21} ${UNS_PRODUCT_8}"
	else
	    if [ "$MKL_IA32_INSTALLED" == "1" ] ; then
		COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED MKL_IA32"
		line="${UNS_PRODUCT_7}"
	    fi
	    if [ "$MKL_INTEL64_INSTALLED" == "1" ] ; then
		COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED MKL_INTEL64"
		line="${UNS_PRODUCT_8}"
	    fi
	fi
	if [ "$MKL_SELECTED" == "1" ] ; then
	    line=" [x] ${INDEX}. $line"
	else
	    line=" [ ] ${INDEX}. $line"
	fi
	echo "$line"
	[ "x$line2" == "x" ] || echo "$line2"
    fi

    if [ "$TBB_INSTALLED" == "1" ] ; then
	INDEX=$(($INDEX+1))
	TBB_INDEX=$INDEX
	COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED TBB"
	line=
	if [ "$TBB_SELECTED" == "1" ] ; then
	    line=" [x] ${INDEX}. ${UNS_PRODUCT_9}"
	else
	    line=" [ ] ${INDEX}. ${UNS_PRODUCT_9}"
	fi
	echo "$line"
    fi

    if [ "$IPP_IA32_INSTALLED" == "1" ] || [ "$IPP_INTEL64_INSTALLED" == "1" ] || [ "$IPP_LP32_INSTALLED" == "1" ] ; then
	INDEX=$(($INDEX+1))
	IPP_INDEX=$INDEX
	IPP_INSTALLED=1
	line2=
	line3=
	line4=
	line5=
	if [ "$IPP_IA32_INSTALLED" == "1" ] && [ "$IPP_INTEL64_INSTALLED" == "1" ] && [ "$IPP_LP32_INSTALLED" == "1" ] ; then
	    COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IPP_IA32 IPP_INTEL64 IPP_LP32"
	    line="${UNS_PRODUCT_10}"
	    line2="            ${UNS_STRING_21} ${UNS_PRODUCT_11}"
	    line3="            ${UNS_STRING_21} ${UNS_PRODUCT_12}"
	else
	    if [ "$IPP_IA32_INSTALLED" == "1" ] ; then
		COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IPP_IA32"
		line="${UNS_PRODUCT_10}"		
	    fi
	    if [ "$IPP_INTEL64_INSTALLED" == "1" ] ; then
		COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IPP_INTEL64"
		line="${UNS_PRODUCT_11}"
	    fi
	    if [ "$IPP_LP32_INSTALLED" == "1" ] ; then
		COMPONENTS_INSTALLED="$COMPONENTS_INSTALLED IPP_LP32"
		line="${UNS_PRODUCT_12}"
	    fi
	    
	    [ "$IPP_IA32_INSTALLED" == "1" ] && [ "$IPP_INTEL64_INSTALLED" == "1" ] && line="${UNS_PRODUCT_10}" && line2="            ${UNS_STRING_21} ${UNS_PRODUCT_11}"
	    [ "$IPP_IA32_INSTALLED" == "1" ] && [ "$IPP_LP32_INSTALLED" == "1" ] && line="${UNS_PRODUCT_10}" && line2="            ${UNS_STRING_21} ${UNS_PRODUCT_12}"
	    [ "$IPP_LP32_INSTALLED" == "1" ] && [ "$IPP_INTEL64_INSTALLED" == "1" ] && line="${UNS_PRODUCT_11}" && line2="            ${UNS_STRING_21} ${UNS_PRODUCT_12}"
	fi
	if [ "$CRYPTO_IPP_IA32_INSTALLED" == "1" ] || [ "$CRYPTO_IPP_INTEL64_INSTALLED" == "1" ] || [ "$CRYPTO_IPP_LP32_INSTALLED" == "1" ] ; then
	    line4="            ${UNS_STRING_21} ${UNS_PRODUCT_13}"
	fi
	if [ "$GEN_IPP_IA32_INSTALLED" == "1" ] || [ "$GEN_IPP_INTEL64_INSTALLED" == "1" ] ; then
	    line5="            ${UNS_STRING_21} ${UNS_PRODUCT_14}"
	fi
	if [ "$THREADED_IPP_IA32_INSTALLED" == "1" ] || [ "$THREADED_IPP_INTEL64_INSTALLED" == "1" ] ; then
	    line6="            ${UNS_STRING_21} ${UNS_PRODUCT_15}"
	fi
	if [ "$IPP_SELECTED" == "1" ] ; then
	    line=" [x] ${INDEX}. $line"
	else
	    line=" [ ] ${INDEX}. $line"
	fi
	echo "$line"
	[ "x$line2" == "x" ] || echo "$line2"
	[ "x$line3" == "x" ] || echo "$line3"
	[ "x$line4" == "x" ] || echo "$line4"
	[ "x$line5" == "x" ] || echo "$line5"
	[ "x$line6" == "x" ] || echo "$line6"
    fi
    
    echo ""
    CHOSEN_INDEX=
    exit_flag=
    while [ 1 -eq 1 ] ; do
	[ "x$exit_flag" == "x1" ] &> /dev/null && break
	echo -n "${UNS_STRING_11}"
	read CHOSEN_INDEX
	if [ "x$CHOSEN_INDEX" == "x" ] &> /dev/null ; then
	    exit_flag_0=1
	    break
	fi
	if [ $CHOSEN_INDEX -eq 0 ] &> /dev/null ; then
	    if [ "x${select_all}" == "x0" ] ; then
		select_all=1
	    else
		select_all=0
	    fi
	    for one in $COMPONENTS_INSTALLED; do
		one=$(echo $one | sed -e"s/_IA32$//g" | sed -e"s/_INTEL64$//g" | sed -e"s/_LP32$//g")
		eval "value=\${${one}_INSTALLED}"
		[ "$value" == "1" ] || continue
		if [ "x${select_all}" == "x0" ] ; then
			eval "${one}_SELECTED="
		else
			eval "${one}_SELECTED=1"
		fi
	    done
	    exit_flag=1
	    break
	fi
	if [ $CHOSEN_INDEX -ge 0 ] &> /dev/null && [ $CHOSEN_INDEX -le $INDEX ] &> /dev/null ; then
	    for one in $COMPONENTS_INSTALLED; do
		one=$(echo $one | sed -e"s/_IA32$//g" | sed -e"s/_INTEL64$//g" | sed -e"s/_LP32$//g")
		eval "value=\${${one}_INSTALLED}"
		[ "$value" == "1" ] || continue
		eval "value=\${${one}_INDEX}"
		if [ "x$value" == "x$CHOSEN_INDEX" ] ; then
		    eval "value=\${${one}_SELECTED}"
		    if [ "x$value" == "x1" ] ; then
			eval "${one}_SELECTED="
		    else
			eval "${one}_SELECTED=1"
		    fi
		    exit_flag=1
		    break
		fi
	    done
	fi
    done
done

fi

if IS_COMMAND_LINE_OPTION_EXISTS silent ; then
    any_silent=
    if IS_COMMAND_LINE_OPTION_EXISTS icc ; then
	IC_SELECTED=1
	any_silent=1
    fi
    if IS_COMMAND_LINE_OPTION_EXISTS ifort ; then
	IF_SELECTED=1
	any_silent=1
    fi
    if IS_COMMAND_LINE_OPTION_EXISTS idb ; then
	IDB_SELECTED=1
	any_silent=1
    fi
    if IS_COMMAND_LINE_OPTION_EXISTS mkl ; then
	MKL_SELECTED=1
	any_silent=1
    fi
    if IS_COMMAND_LINE_OPTION_EXISTS ipp ; then
	IPP_SELECTED=1
	CRYPTO_IPP_SELECTED=1        
        GEN_IPP_SELECTED=1
        THREADED_IPP_SELECTED=1
	any_silent=1
    fi
    if IS_COMMAND_LINE_OPTION_EXISTS tbb ; then
	TBB_SELECTED=1
	any_silent=1
    fi
    if [ "x${any_silent}" == "x" ] ; then
	IC_SELECTED=1
	IF_SELECTED=1
	IDB_SELECTED=1
	MKL_SELECTED=1
	IPP_SELECTED=1
        CRYPTO_IPP_SELECTED=1
        GEN_IPP_SELECTED=1
        THREADED_IPP_SELECTED=1
    	TBB_SELECTED=1    
    fi
fi

for one in $COMPONENTS_INSTALLED ; do
    two=$(echo $one | sed -e"s/_IA32$//g" | sed -e"s/_INTEL64$//g" | sed -e"s/_LP32$//g")
    eval "value=\${${two}_SELECTED}"
    if [ "x$value" == "x1" ] ; then
	COMPONENTS_TO_UNINSTALL="$COMPONENTS_TO_UNINSTALL $one"
	if [ "$one" == "IPP_IA32" ] || [ "$one" == "IPP_INTEL64" ] || [ "$one" == "IPP_LP32" ] ; then
	    eval "value=\${CRYPTO_${one}_INSTALLED}"
	    if [ "$value" == "1" ] ; then
		COMPONENTS_TO_UNINSTALL="$COMPONENTS_TO_UNINSTALL CRYPTO_${one}"
	    fi

	    eval "value=\${GEN_${one}_INSTALLED}"
	    if [ "$value" == "1" ] ; then
		COMPONENTS_TO_UNINSTALL="$COMPONENTS_TO_UNINSTALL GEN_${one}"
	    fi
	    
	    eval "value=\${THREADED_${one}_INSTALLED}"
	    if [ "$value" == "1" ] ; then
		COMPONENTS_TO_UNINSTALL="$COMPONENTS_TO_UNINSTALL THREADED_${one}"
	    fi
	fi
    else
	COMPONENTS_TO_KEEP="$COMPONENTS_TO_KEEP $one"
	if [ "$one" == "IPP_IA32" ] || [ "$one" == "IPP_INTEL64" ] || [ "$one" == "IPP_LP32" ] ; then
	    eval "value=\${CRYPTO_${one}_INSTALLED}"
	    if [ "$value" == "1" ] ; then
		COMPONENTS_TO_KEEP="$COMPONENTS_TO_KEEP CRYPTO_${one}"
	    fi

	    eval "value=\${GEN_${one}_INSTALLED}"
	    if [ "$value" == "1" ] ; then
		COMPONENTS_TO_KEEP="$COMPONENTS_TO_KEEP GEN_${one}"
	    fi
	    
	    eval "value=\${THREADED_${one}_INSTALLED}"
	    if [ "$value" == "1" ] ; then
		COMPONENTS_TO_KEEP="$COMPONENTS_TO_KEEP THREADED_${one}"
	    fi

	fi    
    fi
done

if [ "x$COMPONENTS_TO_UNINSTALL" == "x" ] ; then
    echo "${UNS_STRING_24}"
    if ! IS_COMMAND_LINE_OPTION_EXISTS silent; then
        SET_DEFAULT_ANSWER "ACCEPTED_TO_EXIT_UNINSTALLATION" "Yes"
        READ_YES_NO_ANSWER "${UNS_STRING_25}" "ACCEPTED_TO_EXIT_UNINSTALLATION"
        ACCEPT_UNINSTALL=$?
    else
        ACCEPT_UNINSTALL=0
    fi
    if [ $ACCEPT_UNINSTALL -eq 0 ] ; then
        rm -f /tmp/${GUID}.composerxe2013.install.semaphore &>/dev/null
        exit 0
    fi
else
     ACCEPT_UNINSTALL=0
fi
done

if ! IS_COMMAND_LINE_OPTION_EXISTS silent; then
    SET_DEFAULT_ANSWER "ACCEPTED_TO_UNINSTALL_PRODUCT" "Yes"
    READ_YES_NO_ANSWER "${UNS_STRING_22}" "ACCEPTED_TO_UNINSTALL_PRODUCT"
    rc=$?
    [ $rc == 1 ] && echo "${UNS_STRING_23}" && exit 1
fi

LOG "COMPONENTS_TO_UNINSTALL: $COMPONENTS_TO_UNINSTALL"
LOG "COMPONENTS_TO_KEEP: $COMPONENTS_TO_KEEP"

IS_COMMAND_LINE_OPTION_EXISTS silent || echo -n "${UNS_STRING_13} .."

for one in $COMPONENTS_TO_UNINSTALL; do
    IS_COMMAND_LINE_OPTION_EXISTS silent || echo -n "."
    eval "value=\${${one}_RPM_PACKAGES}"
    packages_unin=$value
    for two in $packages_unin ; do
	found=
	for three in $COMPONENTS_TO_KEEP ; do
	    eval "value=\${${three}_RPM_PACKAGES}"
	    packages_keep=$value
	    for four in $packages_keep ; do
		if [ "$two" == "$four" ] ; then
		    found=1
		    break
		fi
	    done
	    [ "$found" == "1" ] && break
	done
	[ "$found" == "1" ] || RPMS_TO_UNINSTALL="$RPMS_TO_UNINSTALL $two"
    done
done

IS_COMMAND_LINE_OPTION_EXISTS silent || echo

LOG ""
LOG "RPMS_TO_UNINSTALL"
LOG $RPMS_TO_UNINSTALL

globalresult=0
reportentries=

IS_COMMAND_LINE_OPTION_EXISTS silent || echo -n "${UNS_STRING_14} .."

	for one in $RPMS_TO_UNINSTALL ; do
	    if IS_RPM_INSTALLED $one ; then
		IS_COMMAND_LINE_OPTION_EXISTS silent || echo -n "."
		LOG "RPM entry $one is detected for uninstall ..."
		UNINSTALL_RPM "$(echo $one | sed -e"s/\.rpm$//g" -e"s/\.[a-z0-9_]*$//g")"
		rc=$?
		case $rc in
		0)
		    ;;
		1)
		    globalresult=1
		    reportentries="RPM: $one $reportentries"
		    ;;
		esac
	    else
		LOG "Discovering RPM entry $one for uninstall. Did not detect it ..."
	    fi
	done

	for one in $RPMS_TO_UNINSTALL ; do
	    if IS_NONRPM_INSTALLED $one ; then
		IS_COMMAND_LINE_OPTION_EXISTS silent || echo -n "."
		LOG "NONRPM entry $one is detected for uninstall ..."
		NONRPM_DB_ENTRY_FIND_BY_RPMFILE $one
		nrprc=$?
		RPM_TO_UNINSTALL=
		if [ $nrprc -eq 0 ] ; then
		    for subj in $RS; do
			[ "x$subj" == "x" ] && continue
			RPM_TO_UNINSTALL="$subj"
			break
		    done
		fi
		if [ "x$RPM_TO_UNINSTALL" == "x" ] ; then
		    LOG "NONRPM entry $one cant get DB record ..."
		    rc=1
		else
		    NONRPM_UNINSTALL_PACKAGE "$RPM_TO_UNINSTALL"
		    rc=$?
		fi
		case $rc in
		0)
		    ;;
		1)
		    globalresult=1
		    reportentries="NONRPM: $one $reportentries"
		    ;;
		esac
	    else
		LOG "Discovering NONRPM entry $one for uninstall. Did not detect it ..."
	    fi
	done
    
    if [ $globalresult -eq 0 ]; then
	IS_COMMAND_LINE_OPTION_EXISTS silent || echo "${UNS_STRING_15}"
    else
	if IS_COMMAND_LINE_OPTION_EXISTS silent ; then
	    echo "${UNS_STRING_16}"
	    echo "${UNS_STRING_17}"
	    echo "$reportentries"
	    echo "${UNS_STRING_18}"
	fi
    fi

    if ! IS_COMMAND_LINE_OPTION_EXISTS silent; then
    	echo -n "${UNS_STRING_4} $SUMMARY ${UNS_STRING_19}"
	if [ "x${LOG_F}" == "x" ] ; then 
	    echo ". ${UNS_STRING_6} \"$UNINSTALL_LOG_FILE\" ${UNS_STRING_7}"
	else
	    echo
	fi
	echo "${UNS_STRING_20}"
	WAIT_ENTER_KEY
    fi

SHARED_LOCATION_BIN="${SHARED_LOCATION}/composer_xe_2013/bin"
if [ -e "$SHARED_LOCATION_BIN/link_install.sh" &>/dev/null ] ; then
    DIR=$(pwd 2>/dev/null)
    cd ${SHARED_LOCATION_BIN} &>/dev/null
    `${SHARED_LOCATION_BIN}/link_install.sh -u -l ${SHARED_LOCATION} &>/dev/null`
    cd ${DIR} &>/dev/null
fi

REMOVE_EMPTY_DIRS "$RMUPPERDIR"

rm -f /tmp/${GUID}.composerxe2013.install.semaphore &>/dev/null

exit 0
