#ifndef inversion_
#define inversion_

#include "exception.hh"

namespace GSL {

class VectorFunction
{
private:

      int n, p;

public:

      VectorFunction (int n_, int p_) : n(n_), p(p_) 
      {
            if (n <= 0 || p <= 0) throw DIM_NEGATIVE; 
      }
      virtual ~VectorFunction () {}

      virtual void Value (double * f, const double * x) = 0;

      int GetN () { return n; }
      int GetP () { return p; }
};

class JacobiFunction
{
private:

      int n, p;

public:

      JacobiFunction (VectorFunction & vf_) : n(vf_.GetN()), p(vf_.GetP()) {}
      virtual ~JacobiFunction () {}

      virtual void Value (double * J, const double * x) = 0;

      int GetN () { return n; }
      int GetP () { return p; }
};

class JacobiVectorFunction
{
private:

      int n, p;

public:

      JacobiVectorFunction (VectorFunction & vf_) : n(vf_.GetN()), p(vf_.GetP()) {}
      virtual ~JacobiVectorFunction () {}

      virtual void Value (double * f, double * J, const double * x) = 0;

      int GetN () { return n; }
      int GetP () { return p; }
};

class NumericalJacobiFunction : public JacobiFunction
{
private:

      static const double default_delta_x;
      double * delta_x;
      VectorFunction & vf;
      int order;

public:

      NumericalJacobiFunction (VectorFunction & vf_, double * delta_x_, int order_ = 1); 
      virtual ~NumericalJacobiFunction () { delete [] delta_x; }

      virtual void Value (double * J, const double * x);
};

class SimpleJacobiVectorFunction : public JacobiVectorFunction
{
private:

      VectorFunction & vf;
      JacobiFunction & jf;

public:

      SimpleJacobiVectorFunction(VectorFunction & vf_, JacobiFunction & jf_) 
            : JacobiVectorFunction (vf_), vf(vf_), jf(jf_) {}
      virtual ~SimpleJacobiVectorFunction () {}

      virtual void Value (double * f, double * J, const double * x);

};

}
#endif
