#include "v_func.hh"
#include <gsl/gsl_deriv.h>


namespace GSL {

const double NumericalJacobiFunction::default_delta_x = 1e-5;

NumericalJacobiFunction::NumericalJacobiFunction (VectorFunction & vf_, double * delta_x_, int order_) 
      : JacobiFunction (vf_), vf(vf_), order(order_) 
{ 
      int i;
      delta_x = new double [GetP()];
      for (i = 0; i < GetP(); i ++)
      {
            if (delta_x_ == 0) delta_x [i] = default_delta_x;
            else delta_x [i] = delta_x_ [i];
      }
}


void NumericalJacobiFunction::Value (double * J, const double * x)
{
      int i, j;
      int n = GetN(), p = GetP();
      double * f_ori, * f_delta, * f3, * f4;
      double * x1, * x2, * x3, * x4;

      f_ori = new double [n];
      f_delta = new double [n];
      f3 = new double [n];
      f4 = new double [n];
      x1 = new double [p];
      x2 = new double [p];
      x3 = new double [p];
      x4 = new double [p];

      switch (order)
      {
      case 1:

            vf.Value (f_ori, x);

            for (i = 0; i < p; i ++)
            {
                  for (j = 0; j < p; j ++) x1[j] = x[j];
                  x1[i] += delta_x[i];
                  vf.Value (f_delta, x1);

                  for (j = 0; j < n; j ++)
                  {
                        J[j * p + i] = (f_delta[j] - f_ori[j]) / delta_x[i];
                  }
            }
            break;

      case 2:

            for (i = 0; i < p; i ++)
            {
                  for (j = 0; j < p; j ++) 
                  {
                        x1[j] = x[j];
                        x2[j] = x[j];
                  }
                  x1[i] -= delta_x[i] * .5;
                  x2[i] += delta_x[i] * .5;

                  vf.Value (f_ori, x1);
                  vf.Value (f_delta, x2);

                  for (j = 0; j < n; j ++)
                  {
                        J[j * p + i] = (f_delta[j] - f_ori[j]) / delta_x[i];
                  }
            }
            break;

      case 4:

            for (i = 0; i < p; i ++)
            {
                  for (j = 0; j < p; j ++) 
                  {
                        x1[j] = x[j];
                        x2[j] = x[j];
                        x3[j] = x[j];
                        x4[j] = x[j];
                  }
                  x1[i] -= delta_x[i];
                  x2[i] -= delta_x[i] * .5;
                  x3[i] += delta_x[i] * .5;
                  x4[i] += delta_x[i];

                  vf.Value (f_ori, x1);
                  vf.Value (f_delta, x2);
                  vf.Value (f3, x3);
                  vf.Value (f4, x4);

                  for (j = 0; j < n; j ++)
                  {
                        J[j * p + i] = ( f_ori[j] - 8. * f_delta[j] + 8. * f3[j] - f4[j] ) / (6. * delta_x[i]);
                  }
            }
            break;

      default: 
            throw ORDER_NOT_IMPLEMENTED;
      }

      delete [] f_ori;
      delete [] f_delta;
      delete [] f3;
      delete [] f4;
      delete [] x1;
      delete [] x2;
      delete [] x3;
      delete [] x4;
}

void SimpleJacobiVectorFunction::Value (double * f, double * J, const double * x)
{
      vf.Value (f, x);
      jf.Value (J, x);
}

}
