#include "ls.hh"
#include <cstdlib>
#include <cstdio>

static const int N = 300, P = 3;
static const int MAXITER = 1000;

using namespace GSL;
using namespace std;

void pexit (const char * msg)
{
      fprintf (stderr, "%s\n", msg);
      exit (1);
}

class LSExampleFunction : public LSVectorFunction
{
public:
      LSExampleFunction (int n, int p) : LSVectorFunction (n, p) {}
      virtual void RealFunction (double * f, const double * x);
};

class LSExampleJacobiFunction : public JacobiFunction
{
public:
      LSExampleJacobiFunction (VectorFunction & vf) : JacobiFunction (vf) {}
      void Value (double * J, const double * x);
};

void LSExampleFunction::RealFunction (double * f, const double * x)
{
      int i;
      for (i = 0; i < N; i ++)
      {
            double t = - 1.0 + 2.0 * i / N;
            f[i] = x[0] * exp (- x[1] * t) + x[2];
      }
}

void LSExampleJacobiFunction::Value (double * J, const double * x)
{
      int i;
      for (i = 0; i < N; i ++)
      {
            double t = - 1.0 + 2.0 * i / N;
            int p = GetP();
            J [i * p + 0] = exp (- x[1] * t);
            J [i * p + 1] = - x[0] * t * exp (- x[1] * t);
            J [i * p + 2] = 1.0;
      }
}

int main ()
{
      int i, j, k;

      double delta_x[P] = {1e-5, 1e-5, 1e-5};
      int diff_order = 2;
      double x_real[P] = {0.4, 0.5, 2};
      double x_init[P] = {1, 1, 1};
      double real[N], y[N];
      const double epsabs = 1e-5;
      const double epsrel = 1e-4;

      LSExampleFunction ls_func (N, P);

      ls_func.RealFunction (real, x_real);      
      for (i = 0; i < N; i ++) y[i] = real[i] + (- 0.025 + 0.05 * rand() / RAND_MAX);
      ls_func.SetY (y);

      NumericalJacobiFunction njf (ls_func, delta_x, diff_order);
      SimpleJacobiVectorFunction njvf (ls_func, njf);
      LSAllFunction nlsa_func (ls_func, njf, njvf);

      LSExampleJacobiFunction ejf (ls_func);
      SimpleJacobiVectorFunction ejvf (ls_func, ejf);
      LSAllFunction elsa_func (ls_func, ejf, ejvf);

      // LSSolver ls (nlsa_func, LMSDER, x_init);
      LSSolver ls (elsa_func, LMSDER, x_init);
      
      double x[P];
      for (i = 0; i < MAXITER; i ++)
      {
            int res = ls.Iterate() ;
            printf ("After iteration %d\n", i + 1);
            ls.GetX(x);
            double r = ls.GetResidual();
            double covar[P*P];
            ls.GetCovariance (epsrel, covar);

            printf ("\tx0 = %lf, x1 = %lf, x2 = %lf, residual = %lf\n", x[0], x[1], x[2], r);
            printf ("Covar. Matrix:\n");
            for (j = 0; j < P; j ++)
            {
                  printf ("x%d-  ", j);
                  for (k = 0; k < P; k ++) printf ("x%d = %lf, ", k, covar[j * P + k]);
                  printf ("\n");
            }
            printf ("\n");

            if (res) pexit("Iteration stopped unexpected.");
            if (ls.TestDelta (epsabs, epsrel)) break;
      }
      if (i == MAXITER) pexit ("Exceed max. number of iteration steps.");
      printf ("Fitting success.\n");

      double fmin[N];
      double fit[N];
      ls.GetF (fmin);
      ls_func.RealFunction (fit, x);      

      FILE * f1;

      f1 = fopen ("ls_test.data", "w");
      for (i = 0; i < N; i ++)
      {
            double t = - 1.0 + 2.0 * i / N;
            fprintf (f1, "%lf %lf %lf %lf %lf %lf\n", t, fit[i], y[i], fmin[i], real[i], fabs(fit[i] - real[i]));
      }
      fclose (f1);

      return 0;
}
