#include "ls.hh"

namespace GSL {

const gsl_multifit_fdfsolver_type * const LMSDER = gsl_multifit_fdfsolver_lmsder;
const gsl_multifit_fdfsolver_type * const LMDER = gsl_multifit_fdfsolver_lmder;

LSVectorFunction::LSVectorFunction (int n, int p) : VectorFunction (n, p) 
{
      y = new double [n];
}

void LSVectorFunction::SetY (double * y_)
{
      int i;
      for (i = 0; i < GetN(); i ++) y[i] = y_[i];
}

void LSVectorFunction::GetY (double * y_)
{
      int i;
      for (i = 0; i < GetN(); i ++) y_[i] = y[i];
}

void LSVectorFunction::Value (double * f, const double * x)
{
      int i;
      RealFunction (f, x);
      for (i = 0; i < GetN(); i ++) f[i] -= y[i];
}

LSAllFunction::LSAllFunction (VectorFunction & vf_, JacobiFunction & jf_, JacobiVectorFunction & jvf_)
      : vf(vf_), jf(jf_), jvf(jvf_)
{
      if (vf.GetN() != jf.GetN() || vf.GetN() != jvf.GetN()) throw DIM_INCONSISTENT;
      if (vf.GetP() != jf.GetP() || vf.GetP() != jvf.GetP()) throw DIM_INCONSISTENT;
}

int f (const gsl_vector * x, void * data, gsl_vector * res)
{
      int i;

      LSAllFunction & laf = *static_cast<LSAllFunction *>(data);
      int p = laf.GetP();
      int n = laf.GetN();

      double * xx = new double [p];
      double * ff = new double [n];

      for (i = 0; i < p; i ++) xx[i] = gsl_vector_get (x, i);
      laf.Value (ff, xx);
      for (i = 0; i < n; i ++) gsl_vector_set (res, i, ff[i]);

      delete [] xx;
      delete [] ff;

      return GSL_SUCCESS;
}

int df (const gsl_vector * x, void * data, gsl_matrix * J)
{
      int i, j;

      LSAllFunction & laf = *static_cast<LSAllFunction *>(data);
      int p = laf.GetP();
      int n = laf.GetN();

      double * xx = new double [p];
      double * jj = new double [n * p];

      for (i = 0; i < p; i ++) xx[i] = gsl_vector_get (x, i);
      laf.Jacobi (jj, xx);
      for (i = 0; i < n; i ++)
      {
            for (j = 0; j < p; j ++) gsl_matrix_set (J, i, j, jj[i * p + j]);
      }

      delete [] xx;
      delete [] jj;

      return GSL_SUCCESS;
}

int fdf (const gsl_vector * x, void * data, gsl_vector * res, gsl_matrix * J)
{
      int i, j;

      LSAllFunction & laf = *static_cast<LSAllFunction *>(data);
      int p = laf.GetP();
      int n = laf.GetN();

      double * xx = new double [p];
      double * ff = new double [n];
      double * jj = new double [n * p];

      for (i = 0; i < p; i ++) xx[i] = gsl_vector_get (x, i);
      laf.JacobiValue (ff, jj, xx);
      for (i = 0; i < n; i ++)
      {
            gsl_vector_set (res, i, ff[i]);
            for (j = 0; j < p; j ++) gsl_matrix_set (J, i, j, jj[i * p + j]);
      }

      delete [] xx;
      delete [] ff;
      delete [] jj;

      return GSL_SUCCESS;
}

LSSolver::LSSolver (LSAllFunction & laf_, const gsl_multifit_fdfsolver_type * s_type, double * x_init)
      : laf(laf_), n (laf.GetN()), p (laf.GetP()), after_iter (false)

{
      int i;

      solver = gsl_multifit_fdfsolver_alloc (s_type, n, p);
      
      gsl_multifit_function_fdf func_ = {f, df, fdf, n, p, static_cast<void *>(&laf)};
      func = func_;

      gsl_vector * v = gsl_vector_alloc (p);
      for (i = 0; i < p; i ++) gsl_vector_set (v, i, x_init[i]);
      gsl_multifit_fdfsolver_set (solver, &func, v);
}

LSSolver::~LSSolver ()
{
      gsl_multifit_fdfsolver_free (solver);
}

int LSSolver::Iterate ()
{
      int res = gsl_multifit_fdfsolver_iterate (solver);
      after_iter = true;
      return res;
}

bool LSSolver::TestDelta (double epsabs, double epsrel)
{
      if (!after_iter) throw TEST_BEFORE_ITER;
      if ( gsl_multifit_test_delta (solver->dx, solver->x, epsabs, epsrel) == GSL_SUCCESS ) return true;
      else return false;
}

bool LSSolver::TestGradient (double epsabs)
{
      if (!after_iter) throw TEST_BEFORE_ITER;

      bool res;
      gsl_matrix * J;
      gsl_vector * f;
      gsl_vector * g;

      J = gsl_matrix_alloc (n, p);
      f = gsl_vector_alloc (n);
      g = gsl_vector_alloc (p);

      ::GSL::fdf (solver->x, &laf, f, J);
      gsl_multifit_gradient (J, f, g);
      if (gsl_multifit_test_gradient (g, epsabs) == GSL_SUCCESS) res = true;
      else res = false;

      gsl_matrix_free (J);
      gsl_vector_free (f);
      gsl_vector_free (g);

      return res;
}

void LSSolver::SetX (double * x)
{
      int i;
      for (i = 0; i < p; i ++) gsl_vector_set (solver->x, i, x[i]);
      after_iter = false;
}

void LSSolver::GetX (double * x)
{
      int i;
      for (i = 0; i < p; i ++) x[i] = gsl_vector_get (solver->x, i);
}

void LSSolver::GetF (double * f)
{
      int i;
      gsl_vector * ff;

      ff = gsl_vector_alloc (n);
      GSL::f (solver->x, &laf, ff);
      for (i = 0; i < n; i ++) f[i] = gsl_vector_get (ff, i);

      gsl_vector_free (ff);
}

void LSSolver::GetGradient (double * gx)
{
      int i;

      gsl_matrix * J;
      gsl_vector * ff;
      gsl_vector * gg;

      J = gsl_matrix_alloc (n, p);
      ff = gsl_vector_alloc (n);
      gg = gsl_vector_alloc (p);

      GSL::fdf (solver->x, &laf, ff, J);
      gsl_multifit_gradient (J, ff, gg);
      for (i = 0; i < p; i ++) gx[i] = gsl_vector_get (gg, i);

      gsl_matrix_free (J);
      gsl_vector_free (ff);
      gsl_vector_free (gg);

}

double LSSolver::GetResidual ()
{
      int i;

      double res = 0;
      double * f = new double [n];
      GetF (f);
      for (i = 0; i < n; i ++) res += f[i] * f[i];

      delete [] f;
      return res;
}

void LSSolver::GetCovariance (double epsrel, double * covar)
{
      int i, j;

      gsl_matrix * J;
      gsl_matrix * covar_;

      J = gsl_matrix_alloc (n, p);
      covar_ = gsl_matrix_alloc (p, p);

      GSL::df (solver->x, &laf, J);

      gsl_multifit_covar (J, epsrel, covar_);
      for (i = 0; i < p; i ++)
      {
            for (j = 0; j < p; j ++) covar [i * p + j] = gsl_matrix_get (covar_, i, j);
      }

      gsl_matrix_free (J);
      gsl_matrix_free (covar_);
}

}
